<?php

namespace Tests\Unit;

use App\Support\SecureHttpClient;
use Tests\TestCase;

class SecureHttpClientTest extends TestCase
{
    /** @test */
    public function ssl_verification_is_enabled_in_production()
    {
        // Set to production environment
        config(['app.env' => 'production']);
        config(['app.debug' => false]);

        $this->assertTrue(SecureHttpClient::getSslVerification());
        $this->assertFalse(SecureHttpClient::shouldDisableSslVerification());
    }

    /** @test */
    public function ssl_verification_is_enabled_in_staging()
    {
        config(['app.env' => 'staging']);
        config(['app.debug' => false]);

        $this->assertTrue(SecureHttpClient::getSslVerification());
        $this->assertFalse(SecureHttpClient::shouldDisableSslVerification());
    }

    /** @test */
    public function ssl_verification_can_be_disabled_in_local_with_debug()
    {
        config(['app.env' => 'local']);
        config(['app.debug' => true]);

        $this->assertFalse(SecureHttpClient::getSslVerification());
        $this->assertTrue(SecureHttpClient::shouldDisableSslVerification());
    }

    /** @test */
    public function ssl_verification_is_enabled_in_local_without_debug()
    {
        config(['app.env' => 'local']);
        config(['app.debug' => false]);

        $this->assertTrue(SecureHttpClient::getSslVerification());
        $this->assertFalse(SecureHttpClient::shouldDisableSslVerification());
    }

    /** @test */
    public function creates_http_client_with_proper_timeout()
    {
        config(['app.env' => 'production']);
        config(['app.debug' => false]);

        $client = SecureHttpClient::create();
        $config = $client->getConfig();

        $this->assertEquals(30, $config['timeout']);
        $this->assertTrue($config['verify']);
    }

    /** @test */
    public function http_options_return_correct_verification_setting()
    {
        config(['app.env' => 'production']);
        config(['app.debug' => false]);

        $options = SecureHttpClient::getHttpOptions();

        $this->assertArrayHasKey('verify', $options);
        $this->assertTrue($options['verify']);
    }

    /** @test */
    public function stream_context_options_secure_in_production()
    {
        config(['app.env' => 'production']);
        config(['app.debug' => false]);

        $options = SecureHttpClient::getStreamContextOptions();

        $this->assertTrue($options['ssl']['verify_peer']);
        $this->assertTrue($options['ssl']['verify_peer_name']);
        $this->assertFalse($options['ssl']['allow_self_signed']);
    }

    /** @test */
    public function stream_context_options_relaxed_in_local()
    {
        config(['app.env' => 'local']);
        config(['app.debug' => true]);

        $options = SecureHttpClient::getStreamContextOptions();

        $this->assertFalse($options['ssl']['verify_peer']);
        $this->assertFalse($options['ssl']['verify_peer_name']);
        $this->assertTrue($options['ssl']['allow_self_signed']);
    }
}
