<?php

namespace Tests\Unit;

use App\Support\Base64Url;
use PHPUnit\Framework\TestCase;

class Base64UrlTest extends TestCase
{
    /** @test */
    public function it_encodes_data_to_url_safe_base64()
    {
        $data = 'Hello, World!';
        $encoded = Base64Url::encode($data);

        // URL-safe base64 should not contain + or /
        $this->assertStringNotContainsString('+', $encoded);
        $this->assertStringNotContainsString('/', $encoded);
        // Should not have padding
        $this->assertStringNotContainsString('=', $encoded);
    }

    /** @test */
    public function it_decodes_url_safe_base64_data()
    {
        $original = 'Hello, World!';
        $encoded = Base64Url::encode($original);
        $decoded = Base64Url::decode($encoded);

        $this->assertEquals($original, $decoded);
    }

    /** @test */
    public function it_handles_binary_data()
    {
        // Generate some binary data
        $binaryData = random_bytes(256);

        $encoded = Base64Url::encode($binaryData);
        $decoded = Base64Url::decode($encoded);

        $this->assertEquals($binaryData, $decoded);
    }

    /** @test */
    public function it_handles_email_mime_content()
    {
        $mimeMessage = "To: test@example.com\r\n";
        $mimeMessage .= "Subject: Test Subject\r\n";
        $mimeMessage .= "Content-Type: text/html; charset=UTF-8\r\n";
        $mimeMessage .= "\r\n";
        $mimeMessage .= "<html><body>Test body</body></html>";

        $encoded = Base64Url::encode($mimeMessage);
        $decoded = Base64Url::decode($encoded);

        $this->assertEquals($mimeMessage, $decoded);
    }

    /** @test */
    public function it_handles_special_characters()
    {
        $specialChars = "Special chars: !@#$%^&*()_+-=[]{}|;':\",./<>?`~";

        $encoded = Base64Url::encode($specialChars);
        $decoded = Base64Url::decode($encoded);

        $this->assertEquals($specialChars, $decoded);
    }

    /** @test */
    public function it_handles_unicode_content()
    {
        $unicodeContent = "Unicode: 你好世界 مرحبا العالم Привет мир 🌍🎉";

        $encoded = Base64Url::encode($unicodeContent);
        $decoded = Base64Url::decode($encoded);

        $this->assertEquals($unicodeContent, $decoded);
    }

    /** @test */
    public function it_handles_empty_string()
    {
        $encoded = Base64Url::encode('');
        $decoded = Base64Url::decode($encoded);

        $this->assertEquals('', $decoded);
    }

    /** @test */
    public function it_replaces_plus_with_minus()
    {
        $data = "\xFB"; // base64: +w==
        $urlSafe = Base64Url::encode($data);

        $this->assertStringContainsString('-', $urlSafe);
        $this->assertStringNotContainsString('+', $urlSafe);
    }

    /** @test */
    public function it_replaces_slash_with_underscore()
    {
        $data = "\xFF"; // base64: /w==
        $urlSafe = Base64Url::encode($data);

        $this->assertStringContainsString('_', $urlSafe);
        $this->assertStringNotContainsString('/', $urlSafe);
    }
}
