<?php

namespace Tests\Feature;

use App\Models\Review;
use App\Models\Tenant;
use App\Models\User;
use App\Models\Location;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Str;
use Tests\TestCase;

class MultiTenantIsolationTest extends TestCase
{
    use RefreshDatabase;

    private static int $tenantSeed = 1;

    private Tenant $tenant1;
    private Tenant $tenant2;
    private User $superAdmin;
    private User $tenantAdmin;
    private User $agent;
    private User $otherTenantUser;

    protected function setUp(): void
    {
        parent::setUp();

        $seed = self::$tenantSeed++;

        // Create tenants
        $this->tenant1 = Tenant::create([
            'name' => 'Tenant 1',
            'slug' => "tenant-1-{$seed}",
            'is_active' => true,
        ]);

        $this->tenant2 = Tenant::create([
            'name' => 'Tenant 2',
            'slug' => "tenant-2-{$seed}",
            'is_active' => true,
        ]);

        // Create users
        $this->superAdmin = User::factory()->create([
            'role' => 'super_admin',
            'tenant_id' => $this->tenant1->id,
        ]);

        $this->tenantAdmin = User::factory()->create([
            'role' => 'tenant_admin',
            'tenant_id' => $this->tenant1->id,
        ]);

        $this->agent = User::factory()->create([
            'role' => 'agent',
            'tenant_id' => $this->tenant1->id,
        ]);

        $this->otherTenantUser = User::factory()->create([
            'role' => 'tenant_admin',
            'tenant_id' => $this->tenant2->id,
        ]);
    }

    /** @test */
    public function super_admin_can_access_all_tenant_data()
    {
        $this->actingAs($this->superAdmin);

        // Create reviews in different tenants
        $review1 = Review::factory()->create(['tenant_id' => $this->tenant1->id]);
        $review2 = Review::factory()->create(['tenant_id' => $this->tenant2->id]);

        // Super admin should see both
        $reviews = Review::all();
        $this->assertCount(2, $reviews);
    }

    /** @test */
    public function tenant_admin_can_only_access_their_tenant_data()
    {
        // Create reviews in different tenants
        Review::factory()->create([
            'user_id' => $this->tenantAdmin->id,
            'tenant_id' => $this->tenant1->id,
        ]);
        Review::factory()->create([
            'user_id' => $this->otherTenantUser->id,
            'tenant_id' => $this->tenant2->id,
        ]);

        $this->actingAs($this->tenantAdmin);

        // Tenant admin should only see their tenant's reviews
        $reviews = Review::all();
        $this->assertCount(1, $reviews);
        $this->assertEquals($this->tenant1->id, $reviews->first()->tenant_id);
    }

    /** @test */
    public function agent_can_only_access_their_own_data()
    {
        // Create reviews for different users in same tenant
        Review::factory()->create([
            'user_id' => $this->agent->id,
            'tenant_id' => $this->tenant1->id,
        ]);
        Review::factory()->create([
            'user_id' => $this->tenantAdmin->id,
            'tenant_id' => $this->tenant1->id,
        ]);

        $this->actingAs($this->agent);

        // Agent should only see their own reviews
        $reviews = Review::all();
        $this->assertCount(1, $reviews);
        $this->assertEquals($this->agent->id, $reviews->first()->user_id);
    }

    /** @test */
    public function users_cannot_access_other_tenant_data()
    {
        // Create review in tenant 2
        Review::factory()->create([
            'user_id' => $this->otherTenantUser->id,
            'tenant_id' => $this->tenant2->id,
        ]);

        $this->actingAs($this->tenantAdmin);

        // Tenant 1 admin should not see tenant 2 reviews
        $reviews = Review::all();
        $this->assertCount(0, $reviews);
    }

    /** @test */
    public function new_records_automatically_get_user_and_tenant_ids()
    {
        $this->actingAs($this->agent);

        $location = Location::factory()->create([
            'user_id' => $this->agent->id,
            'tenant_id' => $this->tenant1->id,
        ]);

        $review = Review::create([
            'location_id' => $location->id,
            'source' => 'test',
            'external_review_id' => (string) Str::uuid(),
            'stars' => 5,
            'lang' => 'en',
            'text' => 'Test review',
            'status' => 'new',
        ]);

        $this->assertEquals($this->agent->id, $review->user_id);
        $this->assertEquals($this->tenant1->id, $review->tenant_id);
    }

    /** @test */
    public function user_roles_return_correct_values()
    {
        $this->assertTrue($this->superAdmin->isSuperAdmin());
        $this->assertFalse($this->superAdmin->isTenantAdmin());

        $this->assertFalse($this->tenantAdmin->isSuperAdmin());
        $this->assertTrue($this->tenantAdmin->isTenantAdmin());

        $this->assertFalse($this->agent->isSuperAdmin());
        $this->assertFalse($this->agent->isTenantAdmin());
        $this->assertFalse($this->agent->isAdmin());
    }
}
