@extends('layouts.app')

@section('head')
<!-- Self-hosted TinyMCE only -->
<script src="{{ url('tinymce/js/tinymce.min.js') }}"></script>
<script>
// Verify TinyMCE loading
window.addEventListener('load', function() {
    if (typeof tinymce !== 'undefined') {
    } else {
        console.error('Failed to load TinyMCE from local file');
    }
});
</script>
@endsection

@section('content')
<div class="min-h-screen bg-gray-50 py-12">
    <div class="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
        <!-- Header -->
        <div class="mb-8">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900">Edit Email Draft</h1>
                    <p class="mt-2 text-gray-600">Review and edit your AI-generated email draft</p>
                </div>
                <div class="flex space-x-3">
                    <a href="{{ route('email.index') }}" 
                       class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18"></path>
                        </svg>
                        Back to Inbox
                    </a>
                </div>
            </div>
        </div>

        <form method="POST" action="{{ route('email.draft.update', $draft) }}">
            @csrf
            @method('PUT')

            <div class="bg-white shadow rounded-lg">
                <div class="px-4 py-5 sm:p-6">

                    <!-- Subject -->
                    <div class="mb-6">
                        <label for="subject" class="block text-sm font-medium text-gray-700">Subject</label>
                        <div class="mt-1">
                            <input type="text" name="subject" id="subject"
                                   value="{{ old('subject', $draft->thread->subject) }}"
                                   class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md @error('subject') border-red-300 @enderror">
                        </div>
                        @error('subject')
                            <p class="mt-2 text-sm text-red-600">{{ $message }}</p>
                        @enderror
                    </div>

                    <!-- Message Content -->
                    <div class="mb-6">
                        <div class="flex items-center justify-between mb-2">
                            <label for="email-editor" class="block text-sm font-medium text-gray-700">Message Content</label>
                            @if($signatures->count() > 0)
                                <div class="flex items-center space-x-2">
                                    <label for="signature-select" class="text-sm text-gray-600">Add Signature:</label>
                                    <select id="signature-select" class="text-sm border-gray-300 rounded-md focus:ring-indigo-500 focus:border-indigo-500">
                                        <option value="">Select a signature...</option>
                                        @foreach($signatures as $signature)
                                            <option value="{{ $signature->id }}" data-html="{{ $signature->signature_html }}">
                                                {{ $signature->signature_name }}
                                            </option>
                                        @endforeach
                                    </select>
                                    <button type="button" onclick="insertSignature()" 
                                            class="px-3 py-1 text-xs bg-indigo-600 text-white rounded hover:bg-indigo-700">
                                        Insert
                                    </button>
                                    <button type="button" onclick="testSignature()" 
                                            class="px-3 py-1 text-xs bg-green-600 text-white rounded hover:bg-green-700">
                                        Test
                                    </button>
                                </div>
                                <script>
                                </script>
                            @else
                                <div class="text-sm text-gray-500">No signatures available</div>
                            @endif
                        </div>
                        <div class="mt-1">
                            <textarea id="email-editor" 
                                      name="draft_html"
                                      rows="10"
                                      class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md @error('draft_html') border-red-300 @enderror"
                                      placeholder="Enter your email reply here...">{{ old('draft_html', $draft->draft_html) }}</textarea>
                        </div>
                        <p class="mt-2 text-sm text-gray-500">
                            Review and edit your email draft. Use the toolbar above the editor for formatting.
                            @if($signatures->count() > 0)
                                You can also insert signatures using the dropdown above.
                            @endif
                        </p>
                        @error('draft_html')
                            <p class="mt-2 text-sm text-red-600">{{ $message }}</p>
                        @enderror
                    </div>

                    <!-- Preview -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700">Preview</label>
                        <div class="mt-1 border border-gray-300 rounded-md p-4 bg-gray-50 min-h-[100px]">
                            <div id="preview" class="prose max-w-none">
                                {!! old('draft_html', $draft->draft_html) !!}
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Action Buttons -->
            <div class="flex justify-between">
                <div class="flex space-x-3">
                    <button type="submit" 
                            class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500">
                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                        </svg>
                        Save Changes
                    </button>
                </div>
                
                <div class="flex space-x-3">
                    <form method="POST" action="{{ route('email.approve', $draft) }}" class="inline">
                        @csrf
                        <button type="submit" 
                                onclick="return confirm('Are you sure you want to send this email?')"
                                class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500">
                            <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8"></path>
                            </svg>
                            Send Email
                        </button>
                    </form>
                    
                    <form method="POST" action="{{ route('email.draft.delete', $draft) }}" class="inline" 
                          onsubmit="return confirm('Are you sure you want to delete this draft? This action cannot be undone.')">
                        @csrf
                        @method('DELETE')
                        <button type="submit"
                                class="inline-flex items-center px-4 py-2 border border-red-300 text-sm font-medium rounded-md text-red-700 bg-white hover:bg-red-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500">
                            <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                            </svg>
                            Delete Draft
                        </button>
                    </form>
                </div>
            </div>
        </form>
    </div>
</div>

<script>
// Global variables for TinyMCE state management
let tinyMCEReady = false;
let signatureQueue = [];
let editorInstance = null;

document.addEventListener('DOMContentLoaded', function() {
    
    // Check if TinyMCE is available
    if (typeof tinymce === 'undefined') {
        setupTextareaFallback();
        return;
    }
    
    
    // Initialize TinyMCE with proper callbacks
    tinymce.init({
        selector: '#email-editor',
        height: 300,
        menubar: false,
        plugins: 'lists link image charmap print preview anchor searchreplace visualblocks code fullscreen insertdatetime media table paste code help wordcount',
        toolbar: 'undo redo | formatselect | bold italic backcolor | alignleft aligncenter alignright alignjustify | bullist numlist outdent indent | removeformat | help',
        setup: function(editor) {
            
            // Store editor instance globally
            editorInstance = editor;
            
            editor.on('change', function() {
                updatePreview();
            });
            
            editor.on('keyup', function() {
                updatePreview();
            });
        },
        init_instance_callback: function(editor) {
            tinyMCEReady = true;
            editorInstance = editor;
            
            // Process any queued signature insertions
            processSignatureQueue();
            
            // Initial preview update
            setTimeout(function() {
                updatePreview();
            }, 100);
            
            // Show success message
            showEditorStatus('TinyMCE editor loaded successfully', 'success');
        }
    }).then(function(editors) {
        // Handle the Promise-based initialization
        if (editors && editors.length > 0) {
            editorInstance = editors[0];
            tinyMCEReady = true;
            
            // Process any queued signature insertions
            processSignatureQueue();
            
            // Show success message
            showEditorStatus('TinyMCE editor loaded successfully', 'success');
        }
    }).catch(function(error) {
        console.error('TinyMCE initialization failed:', error);
        showEditorStatus('TinyMCE initialization failed: ' + error.message, 'error');
        setupTextareaFallback();
    });
});

function setupTextareaFallback() {
    const textarea = document.getElementById('email-editor');
    if (textarea) {
        textarea.addEventListener('input', updatePreview);
        textarea.addEventListener('keyup', updatePreview);
        showEditorStatus('Using textarea fallback - TinyMCE not available', 'warning');
    }
}

function processSignatureQueue() {
    while (signatureQueue.length > 0) {
        const signatureData = signatureQueue.shift();
        insertSignatureToEditor(signatureData);
    }
}

function showEditorStatus(message, type) {
    // Create or update status message
    let statusDiv = document.getElementById('editor-status');
    if (!statusDiv) {
        statusDiv = document.createElement('div');
        statusDiv.id = 'editor-status';
        statusDiv.style.cssText = 'padding: 8px; margin: 10px 0; border-radius: 4px; font-size: 14px;';
        document.querySelector('#email-editor').parentNode.insertBefore(statusDiv, document.querySelector('#email-editor'));
    }
    
    statusDiv.textContent = message;
    statusDiv.className = type === 'success' ? 'bg-green-100 text-green-800 border border-green-200' : 
                         type === 'warning' ? 'bg-yellow-100 text-yellow-800 border border-yellow-200' : 
                         'bg-red-100 text-red-800 border border-red-200';
    
    // Auto-hide success messages after 3 seconds
    if (type === 'success') {
        setTimeout(() => {
            if (statusDiv) statusDiv.style.display = 'none';
        }, 3000);
    }
}

function updatePreview() {
    const preview = document.getElementById('preview');
    
    if (preview) {
        if (tinyMCEReady && editorInstance) {
            // Use TinyMCE content
            try {
                preview.innerHTML = editorInstance.getContent();
            } catch (error) {
                console.error('Error getting TinyMCE content:', error);
                // Fallback to textarea
                const textarea = document.getElementById('email-editor');
                if (textarea) {
                    preview.innerHTML = textarea.value;
                }
            }
        } else {
            // Fallback to textarea
            const textarea = document.getElementById('email-editor');
            if (textarea) {
                preview.innerHTML = textarea.value;
            }
        }
    }
}

function insertSignature() {
    
    const signatureSelect = document.getElementById('signature-select');
    if (!signatureSelect) {
        console.error('Signature select element not found');
        showEditorStatus('Signature selector not found', 'error');
        return;
    }
    
    const selectedOption = signatureSelect.options[signatureSelect.selectedIndex];
    
    if (!selectedOption.value) {
        showEditorStatus('Please select a signature first', 'warning');
        return;
    }
    
    const signatureHtml = selectedOption.getAttribute('data-html');
    
    if (!signatureHtml) {
        console.error('No signature HTML found');
        showEditorStatus('Signature content not found', 'error');
        return;
    }
    
    // Prepare signature data
    const signatureData = {
        html: signatureHtml,
        timestamp: Date.now()
    };
    
    // Try to insert immediately if TinyMCE is ready
    if (tinyMCEReady && editorInstance) {
        try {
            insertSignatureToEditor(signatureData);
            showEditorStatus('Signature inserted successfully', 'success');
        } catch (error) {
            console.error('Failed to insert signature:', error);
            showEditorStatus('Failed to insert signature: ' + error.message, 'error');
        }
    } else {
        // Queue the signature insertion
        signatureQueue.push(signatureData);
        showEditorStatus('Signature queued - will be inserted when editor is ready', 'warning');
    }
    
    // Reset selection
    signatureSelect.selectedIndex = 0;
}

function insertSignatureToEditor(signatureData) {
    
    if (!editorInstance) {
        throw new Error('Editor instance not available');
    }
    
    // Debug: Check what methods are available
    
    try {
        // Get current content
        const currentContent = editorInstance.getContent();
        
        // Insert signature at the end of the content using TinyMCE's setContent method
        const newContent = currentContent + '<br><br>' + signatureData.html;
        editorInstance.setContent(newContent);
        
        // Update preview
        updatePreview();
        
    } catch (error) {
        console.error('Error inserting signature to TinyMCE:', error);
        
        // Fallback to textarea
        const textarea = document.getElementById('email-editor');
        if (textarea) {
            const currentContent = textarea.value;
            const newContent = currentContent + '\n\n' + signatureData.html;
            textarea.value = newContent;
            updatePreview();
        } else {
            throw new Error('Neither TinyMCE nor textarea found');
        }
    }
}

function testSignature() {
    
    // Test TinyMCE status
    if (tinyMCEReady && editorInstance) {
        showEditorStatus('TinyMCE editor is ready and available', 'success');
    } else {
        showEditorStatus('TinyMCE editor is not ready yet', 'warning');
    }
    
    // Test signature selector
    const signatureSelect = document.getElementById('signature-select');
    if (signatureSelect) {
        showEditorStatus('Signature selector found with ' + signatureSelect.options.length + ' options', 'success');
    } else {
        showEditorStatus('Signature selector not found', 'error');
    }
}
</script>
@endsection
