<x-app-layout>
    <x-slot name="header">
        <h2 class="font-semibold text-xl text-gray-800 leading-tight">
            {{ __('AI Provider Settings') }}
        </h2>
    </x-slot>

    <div class="py-12">
        <div class="max-w-7xl mx-auto sm:px-6 lg:px-8">
            <div class="bg-white overflow-hidden shadow-sm sm:rounded-lg">
                <div class="p-6 text-gray-900">
                    <div class="mb-6">
                        <h3 class="text-lg font-medium text-gray-900 mb-2">AI Provider Configuration</h3>
                        <p class="text-sm text-gray-600">
                            Configure which AI providers to use for generating email and review replies. 
                            Gemini is set as the default provider.
                        </p>
                    </div>

                    <!-- Provider Status Cards -->
                    @php
                        $hasDefaultProvider = collect($providers)->contains('is_default', true);
                    @endphp
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
                        @foreach($providers as $key => $provider)
                            <div class="border rounded-lg p-4 {{ $provider['is_default'] ? 'border-blue-500 bg-blue-50' : 'border-gray-200' }}">
                                <div class="flex items-center justify-between mb-3">
                                    <h4 class="font-medium text-gray-900">{{ $provider['name'] }}</h4>
                                    <div class="flex items-center space-x-2">
                                        @if($provider['is_default'])
                                            <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                                                Default
                                            </span>
                                        @endif
                                        <div class="flex items-center">
                                            <div class="w-2 h-2 rounded-full {{ $provider['available'] && $provider['configured'] ? 'bg-green-500' : 'bg-red-500' }}"></div>
                                            <span class="ml-1 text-xs text-gray-600">
                                                {{ $provider['available'] && $provider['configured'] ? 'Ready' : 'Not Ready' }}
                                            </span>
                                        </div>
                                    </div>
                                </div>
                                
                                <p class="text-sm text-gray-600 mb-3">{{ $provider['description'] }}</p>
                                
                                <div class="flex items-center justify-between">
                                    <label class="flex items-center">
                                        <input type="radio"
                                               name="ai-provider"
                                               class="provider-radio text-blue-600 border-gray-300 focus:ring-blue-200"
                                               data-provider="{{ $key }}"
                                               {{ ($provider['is_default'] || (!$hasDefaultProvider && $provider['enabled'])) ? 'checked' : '' }}
                                               {{ !$provider['available'] ? 'disabled' : '' }}>
                                        <span class="ml-2 text-sm text-gray-700">Enable</span>
                                    </label>
                                    
                                    <button type="button" 
                                            class="test-provider-btn text-sm text-blue-600 hover:text-blue-800 disabled:text-gray-400"
                                            data-provider="{{ $key }}"
                                            {{ !$provider['available'] ? 'disabled' : '' }}>
                                        Test Connection
                                    </button>
                                </div>
                            </div>
                        @endforeach
                    </div>

                    <!-- Default Provider Selection -->
                    <div class="mb-6">
                        <label for="default-provider" class="block text-sm font-medium text-gray-700 mb-2">
                            Default AI Provider
                        </label>
                        <select id="default-provider" 
                                class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-300 focus:ring focus:ring-blue-200 focus:ring-opacity-50">
                            @foreach($providers as $key => $provider)
                                <option value="{{ $key }}" {{ $provider['is_default'] ? 'selected' : '' }}>
                                    {{ $provider['name'] }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <!-- Action Buttons -->
                    <div class="flex items-center space-x-4">
                        <button type="button" 
                                id="test-all-btn"
                                class="inline-flex items-center px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                            <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                            Test All Providers
                        </button>
                        
                        <button type="button" 
                                id="save-settings-btn"
                                class="inline-flex items-center px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                            <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                            </svg>
                            Save Settings
                        </button>
                    </div>

                    <!-- Test Results -->
                    <div id="test-results" class="mt-6 hidden">
                        <h4 class="text-lg font-medium text-gray-900 mb-3">Test Results</h4>
                        <div id="test-results-content" class="space-y-3"></div>
                    </div>

                    <!-- Environment Configuration Help -->
                    <div class="mt-8 p-4 bg-gray-50 rounded-lg">
                        <h4 class="text-sm font-medium text-gray-900 mb-2">Environment Configuration</h4>
                        <p class="text-sm text-gray-600 mb-3">
                            API keys must be set in your <code class="bg-gray-200 px-1 rounded">.env</code> file. Provider toggles and the default provider are saved here in the app.
                        </p>
                        <div class="bg-gray-800 text-green-400 p-3 rounded text-sm font-mono overflow-x-auto">
                            <div># AI Provider API Keys</div>
                            <div>GEMINI_API_KEY=your_gemini_api_key_here</div>
                            <div>OPENAI_API_KEY=your_openai_api_key_here</div>
                            <div>ANTHROPIC_API_KEY=your_anthropic_api_key_here</div>
                            <div></div>
                            <div># Optional global default (fallback)</div>
                            <div>AI_DEFAULT_PROVIDER=gemini</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Loading Overlay -->
    <div id="loading-overlay" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden z-50">
        <div class="flex items-center justify-center h-full">
            <div class="bg-white rounded-lg p-6 max-w-sm w-full mx-4">
                <div class="flex items-center">
                    <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
                    <span class="ml-3 text-gray-700">Testing connection...</span>
                </div>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const testProviderBtns = document.querySelectorAll('.test-provider-btn');
            const testAllBtn = document.getElementById('test-all-btn');
            const saveSettingsBtn = document.getElementById('save-settings-btn');
            const loadingOverlay = document.getElementById('loading-overlay');
            const testResults = document.getElementById('test-results');
            const testResultsContent = document.getElementById('test-results-content');
            const defaultProviderSelect = document.getElementById('default-provider');
            const providerRadios = document.querySelectorAll('.provider-radio');

            // Test individual provider
            testProviderBtns.forEach(btn => {
                btn.addEventListener('click', function() {
                    const provider = this.dataset.provider;
                    testProvider(provider);
                });
            });

            // Test all providers
            testAllBtn.addEventListener('click', function() {
                testAllProviders();
            });

            // Sync default provider when selecting a provider
            providerRadios.forEach(radio => {
                radio.addEventListener('change', function() {
                    if (this.checked) {
                        defaultProviderSelect.value = this.dataset.provider;
                    }
                });
            });

            // Save settings
            saveSettingsBtn.addEventListener('click', function() {
                saveSettings();
            });

            function testProvider(provider) {
                showLoading();
                
                fetch(`/ai/test-provider`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    },
                    body: JSON.stringify({ provider: provider })
                })
                .then(response => response.json())
                .then(data => {
                    hideLoading();
                    showTestResult(provider, data);
                })
                .catch(error => {
                    hideLoading();
                    showTestResult(provider, { success: false, message: 'Network error: ' + error.message });
                });
            }

            function testAllProviders() {
                showLoading();
                
                fetch('/ai/test-all-providers')
                .then(response => response.json())
                .then(data => {
                    hideLoading();
                    showAllTestResults(data.results);
                })
                .catch(error => {
                    hideLoading();
                    showAllTestResults({ error: 'Network error: ' + error.message });
                });
            }

            function saveSettings() {
                const enabledProviders = {};
                let selectedProvider = defaultProviderSelect.value;

                providerRadios.forEach(radio => {
                    if (radio.checked) {
                        selectedProvider = radio.dataset.provider;
                    }
                });

                providerRadios.forEach(radio => {
                    enabledProviders[radio.dataset.provider] = {
                        enabled: radio.dataset.provider === selectedProvider
                    };
                });

                showLoading();

                fetch('/ai/settings', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    },
                    body: JSON.stringify({
                        default_provider: selectedProvider,
                        providers: enabledProviders
                    })
                })
                .then(response => response.json())
                .then(data => {
                    hideLoading();
                        if (data.success) {
                            alert('Settings saved successfully!');
                    } else {
                        alert('Failed to save settings: ' + data.message);
                    }
                })
                .catch(error => {
                    hideLoading();
                    alert('Network error: ' + error.message);
                });
            }

            function getResultMessage(result) {
                if (result.details && result.details.error) {
                    return result.details.error;
                }

                if (result.details && result.details.message) {
                    return result.details.message;
                }

                if (result.error) {
                    return result.error;
                }

                return result.message || '';
            }

            function showTestResult(provider, result) {
                testResults.classList.remove('hidden');
                const detailMessage = getResultMessage(result);
                
                const resultHtml = `
                    <div class="p-3 rounded ${result.success ? 'bg-green-50 border border-green-200' : 'bg-red-50 border border-red-200'}">
                        <div class="flex items-center">
                            <div class="w-2 h-2 rounded-full ${result.success ? 'bg-green-500' : 'bg-red-500'} mr-2"></div>
                            <span class="font-medium">${provider.toUpperCase()}</span>
                            <span class="ml-2 text-sm ${result.success ? 'text-green-700' : 'text-red-700'}">
                                ${result.success ? 'Connection successful' : 'Connection failed'}
                            </span>
                        </div>
                        ${detailMessage ? `<p class="text-sm mt-1 ${result.success ? 'text-green-600' : 'text-red-600'}">${detailMessage}</p>` : ''}
                    </div>
                `;
                
                testResultsContent.innerHTML = resultHtml;
            }

            function showAllTestResults(results) {
                testResults.classList.remove('hidden');
                
                let resultsHtml = '';
                
                if (results.error) {
                    resultsHtml = `
                        <div class="p-3 rounded bg-red-50 border border-red-200">
                            <div class="flex items-center">
                                <div class="w-2 h-2 rounded-full bg-red-500 mr-2"></div>
                                <span class="font-medium text-red-700">Error</span>
                            </div>
                            <p class="text-sm mt-1 text-red-600">${results.error}</p>
                        </div>
                    `;
                } else {
                    Object.entries(results).forEach(([provider, result]) => {
                        const detailMessage = getResultMessage(result);
                        resultsHtml += `
                            <div class="p-3 rounded ${result.success ? 'bg-green-50 border border-green-200' : 'bg-red-50 border border-red-200'}">
                                <div class="flex items-center">
                                    <div class="w-2 h-2 rounded-full ${result.success ? 'bg-green-500' : 'bg-red-500'} mr-2"></div>
                                    <span class="font-medium">${provider.toUpperCase()}</span>
                                    <span class="ml-2 text-sm ${result.success ? 'text-green-700' : 'text-red-700'}">
                                        ${result.success ? 'Connection successful' : 'Connection failed'}
                                    </span>
                                </div>
                                ${detailMessage ? `<p class="text-sm mt-1 ${result.success ? 'text-green-600' : 'text-red-600'}">${detailMessage}</p>` : ''}
                            </div>
                        `;
                    });
                }
                
                testResultsContent.innerHTML = resultsHtml;
            }

            function showLoading() {
                loadingOverlay.classList.remove('hidden');
            }

            function hideLoading() {
                loadingOverlay.classList.add('hidden');
            }
        });
    </script>
</x-app-layout>
