/**
 * TinyMCE Self-Hosted Minimal Editor
 * No CDN, no API keys, basic rich text functionality only
 */
(function(global) {
  'use strict';

  // Create contentEditable div editor
  function createEditor(target, config) {
    const element = document.querySelector(target);
    if (!element) return null;

    // Hide original textarea
    element.style.display = 'none';

    // Create editor container
    const editorContainer = document.createElement('div');
    editorContainer.className = 'tinymce-minimal-editor';
    editorContainer.style.cssText = `
      border: 1px solid #ccc;
      border-radius: 4px;
      min-height: 200px;
      padding: 15px;
      font-family: Arial, sans-serif;
      background: white;
      outline: none;
      line-height: 1.6;
    `;
    
    // Add CSS for proper paragraph spacing
    const styles = document.createElement('style');
    styles.textContent = `
      .tinymce-minimal-editor p {
        margin: 0 0 12px 0;
      }
      .tinymce-minimal-editor p:last-child {
        margin-bottom: 0;
      }
      .tinymce-minimal-editor br {
        line-height: 1.6;
      }
    `;
    document.head.appendChild(styles);

    // Create toolbar
    const toolbar = document.createElement('div');
    toolbar.className = 'tinymce-toolbar';
    toolbar.style.cssText = `
      border: 1px solid #ccc;
      border-bottom: none;
      background: #f8f9fa;
      padding: 8px;
      display: flex;
      gap: 4px;
      align-items: center;
    `;

    // Toolbar buttons
    const buttons = [
      { name: 'Bold', cmd: 'bold', icon: '𝐁' },
      { name: 'Italic', cmd: 'italic', icon: '𝐈' },
      { name: 'Underline', cmd: 'underline', icon: '𝐔' },
      { name: 'Paragraph', cmd: 'formatBlock', cmdArg: 'p', icon: '¶' }
    ];

    buttons.forEach(btn => {
      const btnEl = document.createElement('button');
      btnEl.type = 'button';
      btnEl.textContent = btn.icon;
      btnEl.title = btn.name;
      btnEl.style.cssText = `
        padding: 6px 12px;
        border: 1px solid #ddd;
        background: white;
        cursor: pointer;
        border-radius: 3px;
        font-weight: bold;
      `;
      
      btnEl.addEventListener('click', () => {
        btnEl.focus();
        if (btn.cmdArg) {
          document.execCommand(btn.cmd, false, btn.cmdArg);
        } else {
          document.execCommand(btn.cmd, false, null);
        }
        updatePreview();
      });

      btnEl.addEventListener('mouseenter', () => {
        btnEl.style.backgroundColor = '#e9ecef';
      });

      btnEl.addEventListener('mouseleave', () => {
        btnEl.style.backgroundColor = 'white';
      });

      toolbar.appendChild(btnEl);
    });

    // Insert editor after textarea
    element.parentNode.insertBefore(toolbar, element.nextSibling);
    element.parentNode.insertBefore(editorContainer, toolbar.nextSibling);

    // Set content and ensure proper paragraph formatting
    let content = element.value;
    // Convert text to proper paragraphs if it doesn't have HTML structure
    if (content && !content.includes('<p>') && !content.includes('<br')) {
      // Split by double newlines or periods followed by spaces
      const paragraphs = content.split(/\n\s*\n|\\. /).filter(p => p.trim());
      content = paragraphs.map(p => `<p>${p.trim()}</p>`).join('');
    }
    editorContainer.innerHTML = content;

    // Make editor content editable
    editorContainer.contentEditable = true;
    
    // Handle Enter key for proper paragraph creation
    editorContainer.addEventListener('keydown', (e) => {
      if (e.key === 'Enter') {
        // Allow default behavior first (creates newline)
        setTimeout(() => {
          // Then format as paragraph
          document.execCommand('formatBlock', false, 'p');
          updatePreview();
        }, 10);
      }
    });

    // Editor API object
    const editor = {
      getContent: () => editorContainer.innerHTML,
      setContent: (content) => {
        editorContainer.innerHTML = content;
        element.value = content; // Keep textarea in sync
      },
      focus: () => editorContainer.focus(),
      destroy: () => {
        element.style.display = '';
        toolbar.remove();
        editorContainer.remove();
      },
      on: (event, callback) => {
        if (event === 'change') {
          editorContainer.addEventListener('input', callback);
        }
      }
    };

    // Sync content changes back to textarea
    editorContainer.addEventListener('input', () => {
      element.value = editorContainer.innerHTML;
      updatePreview();
    });

    // Update preview function
    window.updatePreview = function() {
      const preview = document.getElementById('preview');
      if (preview) {
        preview.innerHTML = editorContainer.innerHTML;
      }
    };

    return editor;
  }

  // TinyMCE API
  const TinyMCE = {
    init: function(config) {
      return new Promise((resolve) => {
        setTimeout(() => {
          const editor = createEditor(config.selector, config);
          resolve([editor || { 
            getContent: () => '', 
            setContent: () => {}, 
            focus: () => {},
            destroy: () => {}
          }]);
        }, 10);
      });
    },

    remove: function(selector) {
      const element = document.querySelector(selector);
      if (element && element.style.display === 'none') {
        element.style.display = '';
        const toolbar = element.nextElementSibling;
        const editor = toolbar ? toolbar.nextElementSibling : null;
        if (toolbar && toolbar.classList.contains('tinymce-toolbar')) toolbar.remove();
        if (editor && editor.classList.contains('tinymce-minimal-editor')) editor.remove();
      }
    }
  };

  // Export to global
  global.tinymce = TinyMCE;
  
})(typeof window !== 'undefined' ? window : this);
