<?php

namespace App\Support;

use Illuminate\Support\Facades\Log;

/**
 * Helper class for retry logic with exponential backoff
 *
 * Use this helper to wrap API calls that may fail due to transient errors
 * such as network issues, rate limiting, or temporary service unavailability.
 */
class RetryHelper
{
    /**
     * Default maximum number of retry attempts
     */
    public const DEFAULT_MAX_ATTEMPTS = 3;

    /**
     * Default base delay in milliseconds
     */
    public const DEFAULT_BASE_DELAY_MS = 1000;

    /**
     * Default maximum delay in milliseconds (cap for exponential backoff)
     */
    public const DEFAULT_MAX_DELAY_MS = 10000;

    /**
     * Execute a callback with retry logic and exponential backoff
     *
     * @param callable $callback The function to execute
     * @param int $maxAttempts Maximum number of attempts (default: 3)
     * @param int $baseDelayMs Base delay in milliseconds (default: 1000)
     * @param int $maxDelayMs Maximum delay cap in milliseconds (default: 10000)
     * @param array $retryableExceptions Exception classes that should trigger retry
     * @return mixed The result of the callback
     * @throws \Exception The last exception if all retries fail
     */
    public static function retry(
        callable $callback,
        int $maxAttempts = self::DEFAULT_MAX_ATTEMPTS,
        int $baseDelayMs = self::DEFAULT_BASE_DELAY_MS,
        int $maxDelayMs = self::DEFAULT_MAX_DELAY_MS,
        array $retryableExceptions = []
    ) {
        $lastException = null;
        $attempt = 0;

        while ($attempt < $maxAttempts) {
            $attempt++;

            try {
                return $callback();
            } catch (\Exception $e) {
                $lastException = $e;

                // Check if this exception type should trigger a retry
                if (!empty($retryableExceptions) && !self::isRetryable($e, $retryableExceptions)) {
                    throw $e;
                }

                // Don't sleep on the last attempt
                if ($attempt < $maxAttempts) {
                    $delay = self::calculateDelay($attempt, $baseDelayMs, $maxDelayMs);

                    if (Log::getFacadeRoot()) {
                        Log::warning("Retry attempt {$attempt}/{$maxAttempts} failed, retrying in {$delay}ms", [
                            'error' => $e->getMessage(),
                            'exception_class' => get_class($e),
                        ]);
                    }

                    usleep($delay * 1000); // Convert ms to microseconds
                }
            }
        }

        if (Log::getFacadeRoot()) {
            Log::error("All {$maxAttempts} retry attempts failed", [
                'error' => $lastException->getMessage(),
                'exception_class' => get_class($lastException),
            ]);
        }

        throw $lastException;
    }

    /**
     * Calculate delay with exponential backoff and jitter
     *
     * @param int $attempt Current attempt number (1-based)
     * @param int $baseDelayMs Base delay in milliseconds
     * @param int $maxDelayMs Maximum delay cap
     * @return int Delay in milliseconds
     */
    public static function calculateDelay(int $attempt, int $baseDelayMs, int $maxDelayMs): int
    {
        // Exponential backoff: base * 2^(attempt-1)
        $exponentialDelay = $baseDelayMs * pow(2, $attempt - 1);

        // Cap at max delay
        $cappedDelay = min($exponentialDelay, $maxDelayMs);

        // Add jitter (random 0-25% of the delay)
        $jitter = rand(0, (int)($cappedDelay * 0.25));

        return (int)($cappedDelay + $jitter);
    }

    /**
     * Check if an exception should trigger a retry
     *
     * @param \Exception $exception
     * @param array $retryableExceptions
     * @return bool
     */
    private static function isRetryable(\Exception $exception, array $retryableExceptions): bool
    {
        foreach ($retryableExceptions as $retryableClass) {
            if ($exception instanceof $retryableClass) {
                return true;
            }
        }

        // Also retry on common transient error patterns
        $message = strtolower($exception->getMessage());
        $transientPatterns = [
            'timeout',
            'timed out',
            'connection reset',
            'connection refused',
            'too many requests',
            'rate limit',
            '429',
            '500',
            '502',
            '503',
            '504',
            'temporarily unavailable',
            'service unavailable',
        ];

        foreach ($transientPatterns as $pattern) {
            if (str_contains($message, $pattern)) {
                return true;
            }
        }

        return false;
    }
}
