<?php

namespace App\Services;

use App\Models\User;
use App\Models\Tenant;
use Illuminate\Support\Facades\Auth;

class UserContextService
{
    /**
     * Get the current user
     */
    public function getCurrentUser(): ?User
    {
        return Auth::user();
    }

    /**
     * Get the current tenant
     */
    public function getCurrentTenant(): ?Tenant
    {
        $user = $this->getCurrentUser();
        
        if (!$user) {
            return null;
        }

        if ($user->isSuperAdmin()) {
            // Super admin can access any tenant
            $tenantId = request()->get('tenant_id') ?? $user->tenant_id;
            return $tenantId ? Tenant::find($tenantId) : null;
        }

        return $user->tenant;
    }

    /**
     * Check if current user can access a specific tenant
     */
    public function canAccessTenant($tenantId): bool
    {
        $user = $this->getCurrentUser();
        
        if (!$user) {
            return false;
        }

        return $user->canAccessTenant($tenantId);
    }

    /**
     * Get user's mail accounts
     */
    public function getUserMailAccounts()
    {
        $user = $this->getCurrentUser();
        
        if (!$user) {
            return \App\Models\MailAccount::whereRaw('1 = 0'); // Empty query builder
        }

        if ($user->isSuperAdmin()) {
            // Super admin can see all mail accounts
            return \App\Models\MailAccount::with(['user', 'tenant']);
        }

        // Tenant admin should only see their own mail accounts
        if ($user->isTenantAdmin()) {
            $query = \App\Models\MailAccount::where('user_id', $user->id);
            if ($user->tenant_id) {
                $query->where('tenant_id', $user->tenant_id);
            }
            return $query->with(['user', 'tenant']);
        }

        // Regular users see only their own mail accounts
        $query = \App\Models\MailAccount::where('user_id', $user->id);
        if ($user->tenant_id) {
            $query->where('tenant_id', $user->tenant_id);
        }
        return $query->with(['user', 'tenant']);
    }

    /**
     * Get user's email threads
     */
    public function getUserThreads()
    {
        $user = $this->getCurrentUser();
        
        if (!$user) {
            return \App\Models\Thread::whereRaw('1 = 0'); // Empty query builder
        }

        if ($user->isSuperAdmin()) {
            // Super admin can see all threads
            return \App\Models\Thread::with(['user', 'tenant', 'mailAccount']);
        }

        // Tenant admin should only see their own threads
        if ($user->isTenantAdmin()) {
            $query = \App\Models\Thread::where('user_id', $user->id);
            if ($user->tenant_id) {
                $query->where('tenant_id', $user->tenant_id);
            }
            return $query->with(['user', 'tenant', 'mailAccount']);
        }

        // Regular users see only their own threads
        if ($user->tenant_id) {
            return \App\Models\Thread::where('user_id', $user->id)
                ->where('tenant_id', $user->tenant_id)
                ->with(['user', 'tenant', 'mailAccount']);
        }
        
        // Super Admin (no tenant_id) sees all threads
        return \App\Models\Thread::with(['user', 'tenant', 'mailAccount']);
    }

    /**
     * Get user's email drafts
     */
    public function getUserEmailDrafts()
    {
        $user = $this->getCurrentUser();
        
        if (!$user) {
            return \App\Models\EmailDraft::whereRaw('1 = 0'); // Empty query builder
        }

        if ($user->isSuperAdmin()) {
            // Super admin can see all drafts
            return \App\Models\EmailDraft::with(['user', 'tenant', 'thread']);
        }

        // Regular users only see their own drafts
        $query = \App\Models\EmailDraft::where('user_id', $user->id);
        if ($user->tenant_id) {
            $query->where('tenant_id', $user->tenant_id);
        }
        return $query->with(['user', 'tenant', 'thread']);
    }

    /**
     * Get user's locations
     */
    public function getUserLocations()
    {
        $user = $this->getCurrentUser();
        
        if (!$user) {
            return \App\Models\Location::whereRaw('1 = 0'); // Empty query builder
        }

        if ($user->isSuperAdmin()) {
            // Super admin can see all locations
            return \App\Models\Location::with(['user', 'tenant']);
        }

        // Regular users only see their own locations
        $query = \App\Models\Location::where('user_id', $user->id);
        if ($user->tenant_id) {
            $query->where('tenant_id', $user->tenant_id);
        }
        return $query->with(['user', 'tenant']);
    }

    /**
     * Create a new record with user context
     */
    public function createWithUserContext($model, $data)
    {
        $user = $this->getCurrentUser();
        $tenant = $this->getCurrentTenant();
        
        if (!$user) {
            throw new \Exception('User not authenticated');
        }

        $data['user_id'] = $user->id;
        $data['tenant_id'] = $tenant?->id;

        return $model::create($data);
    }

    /**
     * Update a record with user context validation
     */
    public function updateWithUserContext($model, $id, $data)
    {
        $user = $this->getCurrentUser();
        
        if (!$user) {
            throw new \Exception('User not authenticated');
        }

        $record = $model::findOrFail($id);

        // Check if user can access this record
        if (!$user->isSuperAdmin() && $record->user_id !== $user->id) {
            throw new \Exception('Unauthorized access to record');
        }

        return $record->update($data);
    }

    /**
     * Delete a record with user context validation
     */
    public function deleteWithUserContext($model, $id)
    {
        $user = $this->getCurrentUser();
        
        if (!$user) {
            throw new \Exception('User not authenticated');
        }

        $record = $model::findOrFail($id);

        // Check if user can access this record
        if (!$user->isSuperAdmin() && $record->user_id !== $user->id) {
            throw new \Exception('Unauthorized access to record');
        }

        return $record->delete();
    }
}
