<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class SecurityService
{
    /**
     * Sanitize HTML content for safe display
     */
    public static function sanitizeHtml(string $html, array $allowedTags = null): string
    {
        $defaultAllowedTags = [
            'p', 'br', 'strong', 'em', 'u', 'b', 'i', 'ul', 'ol', 'li',
            'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'blockquote', 'pre',
            'a', 'span', 'div'
        ];
        
        $allowedTags = $allowedTags ?? $defaultAllowedTags;
        $allowedTagsString = '<' . implode('><', $allowedTags) . '>';
        
        // Remove dangerous attributes
        $html = preg_replace('/\s*on\w+\s*=\s*["\'][^"\']*["\']/', '', $html);
        $html = preg_replace('/\s*javascript\s*:/i', '', $html);
        $html = preg_replace('/\s*vbscript\s*:/i', '', $html);
        $html = preg_replace('/\s*data\s*:/i', '', $html);
        
        // Strip tags except allowed ones
        $html = strip_tags($html, $allowedTagsString);
        
        // Clean up any remaining dangerous content
        $html = self::removeDangerousContent($html);
        
        return $html;
    }

    /**
     * Remove dangerous content from HTML
     */
    private static function removeDangerousContent(string $html): string
    {
        // Remove script tags and their content
        $html = preg_replace('/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/mi', '', $html);
        
        // Remove style tags and their content
        $html = preg_replace('/<style\b[^<]*(?:(?!<\/style>)<[^<]*)*<\/style>/mi', '', $html);
        
        // Remove iframe tags
        $html = preg_replace('/<iframe\b[^>]*>.*?<\/iframe>/mi', '', $html);
        
        // Remove object and embed tags
        $html = preg_replace('/<(object|embed)\b[^>]*>.*?<\/(object|embed)>/mi', '', $html);
        
        // Remove form tags
        $html = preg_replace('/<form\b[^>]*>.*?<\/form>/mi', '', $html);
        
        // Remove input tags
        $html = preg_replace('/<input\b[^>]*>/mi', '', $html);
        
        // Remove dangerous attributes from remaining tags
        $html = preg_replace('/\s*(on\w+|javascript|vbscript|data)\s*=\s*["\'][^"\']*["\']/', '', $html);
        
        return $html;
    }

    /**
     * Sanitize user input for database storage
     */
    public static function sanitizeInput(string $input): string
    {
        // Remove null bytes
        $input = str_replace("\0", '', $input);
        
        // Normalize line endings
        $input = str_replace(["\r\n", "\r"], "\n", $input);
        
        // Trim whitespace
        $input = trim($input);
        
        // Remove excessive whitespace
        $input = preg_replace('/\s+/', ' ', $input);
        
        return $input;
    }

    /**
     * Validate and sanitize email content
     */
    public static function sanitizeEmailContent(string $content): string
    {
        // First sanitize HTML
        $content = self::sanitizeHtml($content);
        
        // Remove any remaining dangerous patterns
        $content = preg_replace('/javascript\s*:/i', '', $content);
        $content = preg_replace('/vbscript\s*:/i', '', $content);
        $content = preg_replace('/data\s*:/i', '', $content);
        
        return $content;
    }

    /**
     * Escape output for safe display
     */
    public static function escapeOutput(string $output): string
    {
        return htmlspecialchars($output, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    }

    /**
     * Validate SQL injection attempts
     */
    public static function detectSqlInjection(string $input): bool
    {
        $sqlPatterns = [
            '/(\bunion\b.*\bselect\b)/i',
            '/(\bselect\b.*\bfrom\b)/i',
            '/(\binsert\b.*\binto\b)/i',
            '/(\bupdate\b.*\bset\b)/i',
            '/(\bdelete\b.*\bfrom\b)/i',
            '/(\bdrop\b.*\btable\b)/i',
            '/(\balter\b.*\btable\b)/i',
            '/(\bcreate\b.*\btable\b)/i',
            '/(\bexec\b|\bexecute\b)/i',
            '/(\bscript\b.*\b>)/i',
            '/(\bwaitfor\b.*\bdelay\b)/i',
            '/(\bxp_cmdshell\b)/i',
            '/(\bsp_executesql\b)/i',
            '/(\b--\b|\b#\b|\b\/\*.*\*\/)/i',
            '/(\bchar\b.*\b\(.*\b\))/i',
            '/(\bconcat\b.*\b\(.*\b\))/i',
            '/(\bversion\b.*\b\(.*\b\))/i',
            '/(\buser\b.*\b\(.*\b\))/i',
            '/(\bdatabase\b.*\b\(.*\b\))/i',
            '/(\bschema\b.*\b\(.*\b\))/i',
        ];

        foreach ($sqlPatterns as $pattern) {
            if (preg_match($pattern, $input)) {
                self::logSecurityThreat('SQL_INJECTION_ATTEMPT', $input);
                return true;
            }
        }

        return false;
    }

    /**
     * Validate XSS attempts
     */
    public static function detectXss(string $input): bool
    {
        $xssPatterns = [
            '/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/mi',
            '/<iframe\b[^>]*>.*?<\/iframe>/mi',
            '/<object\b[^>]*>.*?<\/object>/mi',
            '/<embed\b[^>]*>.*?<\/embed>/mi',
            '/<link\b[^>]*>/mi',
            '/<meta\b[^>]*>/mi',
            '/<style\b[^>]*>.*?<\/style>/mi',
            '/<form\b[^>]*>.*?<\/form>/mi',
            '/<input\b[^>]*>/mi',
            '/<button\b[^>]*>.*?<\/button>/mi',
            '/<select\b[^>]*>.*?<\/select>/mi',
            '/<textarea\b[^>]*>.*?<\/textarea>/mi',
            '/on\w+\s*=\s*["\'][^"\']*["\']/i',
            '/javascript\s*:/i',
            '/vbscript\s*:/i',
            '/data\s*:/i',
            '/<img\b[^>]*onerror\s*=/i',
            '/<svg\b[^>]*onload\s*=/i',
        ];

        foreach ($xssPatterns as $pattern) {
            if (preg_match($pattern, $input)) {
                self::logSecurityThreat('XSS_ATTEMPT', $input);
                return true;
            }
        }

        return false;
    }

    /**
     * Log security threats
     */
    private static function logSecurityThreat(string $type, string $input): void
    {
        Log::warning('Security threat detected', [
            'type' => $type,
            'input' => Str::limit($input, 500),
            'ip' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'user_id' => auth()->id(),
            'url' => request()->fullUrl(),
            'timestamp' => now()->toISOString(),
        ]);
    }

    /**
     * Get safe HTML for display
     */
    public static function getSafeHtml(string $html, array $allowedTags = null): string
    {
        if (self::detectXss($html)) {
            return self::escapeOutput($html);
        }
        
        return self::sanitizeHtml($html, $allowedTags);
    }

    /**
     * Validate file upload security
     */
    public static function validateFileUpload($file): array
    {
        $errors = [];
        
        if (!$file) {
            $errors[] = 'No file provided';
            return $errors;
        }

        // Check file size (max 10MB)
        if ($file->getSize() > 10 * 1024 * 1024) {
            $errors[] = 'File size too large (max 10MB)';
        }

        // Check file extension
        $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'pdf', 'doc', 'docx', 'txt'];
        $extension = strtolower($file->getClientOriginalExtension());
        
        if (!in_array($extension, $allowedExtensions)) {
            $errors[] = 'File type not allowed';
        }

        // Check MIME type
        $allowedMimeTypes = [
            'image/jpeg', 'image/png', 'image/gif',
            'application/pdf', 'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'text/plain'
        ];
        
        if (!in_array($file->getMimeType(), $allowedMimeTypes)) {
            $errors[] = 'File MIME type not allowed';
        }

        // Check for malicious content in file name
        if (self::detectXss($file->getClientOriginalName())) {
            $errors[] = 'Suspicious file name detected';
        }

        return $errors;
    }

    /**
     * Generate secure random string
     */
    public static function generateSecureToken(int $length = 32): string
    {
        return bin2hex(random_bytes($length));
    }

    /**
     * Validate password strength
     */
    public static function validatePasswordStrength(string $password): array
    {
        $errors = [];
        
        if (strlen($password) < 8) {
            $errors[] = 'Password must be at least 8 characters long';
        }
        
        if (!preg_match('/[A-Z]/', $password)) {
            $errors[] = 'Password must contain at least one uppercase letter';
        }
        
        if (!preg_match('/[a-z]/', $password)) {
            $errors[] = 'Password must contain at least one lowercase letter';
        }
        
        if (!preg_match('/[0-9]/', $password)) {
            $errors[] = 'Password must contain at least one number';
        }
        
        if (!preg_match('/[^A-Za-z0-9]/', $password)) {
            $errors[] = 'Password must contain at least one special character';
        }
        
        return $errors;
    }
}
