<?php

namespace App\Services;

use App\Models\MailAccount;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Mail\Message;
use Illuminate\Support\Facades\Validator;
use App\Support\SecureHttpClient;

class SMTPEmailService
{
    /**
     * Send email via SMTP
     * 
     * @param MailAccount $account SMTP mail account
     * @param array $messageData Email message data
     * @return array Result with success status and message
     */
    public function sendEmail(MailAccount $account, array $messageData): array
    {
        try {
            // Validation
            $validator = Validator::make($messageData, [
                'to' => 'required|email',
                'subject' => 'required|string',
                'body' => 'required|string',
                'thread_id' => 'nullable|string',
                'in_reply_to' => 'nullable|string',
                'references' => 'nullable|string',
            ]);

            if ($validator->fails()) {
                return [
                    'success' => false,
                    'message' => 'Invalid email data: ' . $validator->errors()->first()
                ];
            }

            // Configure SMTP settings
            $config = $this->configureSMTP($account);

            // Create a custom mailer with the account's SMTP settings
            $mailer = app('mail.manager')->mailer('smtp');

            // Send raw email with custom headers
            $mailer->raw($messageData['body'], function (Message $message) use ($messageData, $account) {
                $message->from($account->email, $account->smtp_from_name ?: $account->from_name)
                        ->to($messageData['to'])
                        ->subject($messageData['subject']);

                // Add threading headers if available
                if (!empty($messageData['thread_id'])) {
                    $message->getHeaders()->addTextHeader('X-Original-Thread-ID', $messageData['thread_id']);
                }
                
                if (!empty($messageData['in_reply_to'])) {
                    $message->getHeaders()->addTextHeader('In-Reply-To', $messageData['in_reply_to']);
                }
                
                if (!empty($messageData['references'])) {
                    $message->getHeaders()->addTextHeader('References', $messageData['references']);
                }

                // Set content type for HTML
                if ($this->isHtml($messageData['body'])) {
                    $message->getHeaders()->addTextHeader('Content-Type', 'text/html; charset=UTF-8');
                }
            });

            Log::info('SMTP email sent successfully', [
                'account_email' => $account->email,
                'to' => $messageData['to'],
                'subject' => $messageData['subject']
            ]);

            return [
                'success' => true,
                'message' => 'Email sent successfully via SMTP',
                'provider' => 'smtp',
                'account' => $account->email
            ];

        } catch (\Exception $e) {
            Log::error('SMTP email sending failed', [
                'account_email' => $account->email,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'message' => 'Failed to send email via SMTP: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Test SMTP connection
     * 
     * @param array $smtpConfig SMTP configuration
     * @return array Test result
     */
    public function testConnection(array $smtpConfig): array
    {
        try {
            // Create a socket connection to test SMTP with environment-aware SSL settings
            $context = stream_context_create(SecureHttpClient::getStreamContextOptions());

            $host = $smtpConfig['host'];
            $port = $smtpConfig['port'];
            $timeout = 10; // config('mail.mailers.smtp.timeout', 10);

            // Try to connect
            $connection = stream_socket_client(
                "tcp://{$host}:{$port}",
                $errno,
                $errstr,
                $timeout,
                STREAM_CLIENT_CONNECT,
                $context
            );

            if (!$connection) {
                return [
                    'success' => false,
                    'message' => "Failed to connect to SMTP server: {$errstr} (Error {$errno})"
                ];
            }

            // Close the connection
            fclose($connection);

            return [
                'success' => true,
                'message' => 'SMTP connection test successful - server is reachable'
            ];

        } catch (\Exception $e) {
            $errorMessage = $e->getMessage();
            
            // Provide more helpful error messages for common issues
            if (strpos($errorMessage, 'Unable to connect') !== false) {
                return [
                    'success' => false,
                    'message' => "❌ Cannot connect to SMTP server: {$host}:{$port}\n\n🔧 Troubleshooting:\n• Check if SMTP host is correct (try smtp.hostinger.com)\n• Verify port number (587 for TLS, 465 for SSL)\n• Check firewall/network settings\n• Ensure SMTP service is enabled with your hosting provider"
                ];
            } elseif (strpos($errorMessage, 'Connection refused') !== false) {
                return [
                    'success' => false,
                    'message' => "❌ SMTP server refused connection (port {$port} blocked)\n\n🔧 Try:\n• Port 465 with SSL encryption\n• Port 587 with TLS encryption\n• Contact your hosting provider if ports are blocked"
                ];
            }
            
            return [
                'success' => false,
                'message' => 'SMTP connection test failed: ' . $errorMessage
            ];
        }
    }

    /**
     * Configure SMTP settings for Laravel Mail
     * 
     * @param MailAccount $account
     * @return array Configuration array
     */
    private function configureSMTP(MailAccount $account): array
    {
        $config = [
            'transport' => 'smtp',
            'host' => $account->smtp_host,
            'port' => $account->smtp_port,
            'encryption' => $account->smtp_encryption ?: 'tls',
            'username' => $account->smtp_username ?: $account->email,
            'password' => $account->smtp_password,
            'timeout' => 30,
            'local_domain' => parse_url(config('app.url'), PHP_URL_HOST),
        ];

        // Update Laravel Mail configuration temporarily
        config([
            'mail.default' => 'smtp',
            'mail.mailers.smtp' => $config
        ]);

        return $config;
    }

    /**
     * Check if content is HTML
     * 
     * @param string $content
     * @return bool
     */
    private function isHtml(string $content): bool
    {
        return strip_tags($content) !== $content;
    }
}
