<?php

namespace App\Services;

use App\Models\Location;
use App\Models\Review;
use Illuminate\Support\Facades\Log;

/**
 * Review Ingestion Service
 *
 * Centralized service for ingesting reviews from Google Business Profile locations.
 * Used by both ReviewController (manual) and IngestReviewsJob (automatic).
 *
 * Reference: AUTO_INGESTION.md Section 4.4
 */
class ReviewIngestionService
{
    private GoogleBusinessProfile $gbpService;

    public function __construct(GoogleBusinessProfile $gbpService)
    {
        $this->gbpService = $gbpService;
    }

    /**
     * Ingest reviews for a specific location
     *
     * @param Location $location
     * @return array Result with success status and counts
     * @throws \Exception
     */
    public function ingestForLocation(Location $location): array
    {
        try {
            // Extract numeric IDs from resource names
            $accountId = basename($location->account_id); // Extract "106708635971562662717" from "accounts/106708635971562662717"
            $locationId = basename($location->location_id); // Extract "13117477215661797539" from "locations/13117477215661797539"

            Log::info('Fetching reviews for location', [
                'location_name' => $location->display_name,
                'account_id' => $accountId,
                'location_id' => $locationId
            ]);

            $reviews = $this->gbpService->listReviews($accountId, $locationId, $location->user_id);

            Log::info('Retrieved reviews from API', [
                'location_name' => $location->display_name,
                'review_count' => count($reviews)
            ]);

            $ingestedCount = 0;
            $newReviews = 0;
            $updatedReviews = 0;

            foreach ($reviews as $reviewData) {
                // Add safety checks for required fields
                if (!isset($reviewData['id']) || !isset($reviewData['stars']) || !isset($reviewData['comment'])) {
                    Log::warning('Skipping review with missing required fields', ['review_data' => $reviewData]);
                    continue;
                }

                // Convert text star ratings to integers
                $starRating = $this->convertStarRatingToInt($reviewData['stars']);
                if ($starRating === null) {
                    Log::warning('Skipping review with invalid star rating', [
                        'review_id' => $reviewData['id'],
                        'stars' => $reviewData['stars']
                    ]);
                    continue;
                }

                $review = Review::updateOrCreate(
                    [
                        'location_id' => $location->id,
                        'external_review_id' => $reviewData['id'],
                    ],
                    [
                        'user_id' => $location->user_id,
                        'tenant_id' => $location->tenant_id,
                        'source' => 'google_gbp',
                        'stars' => $starRating,
                        'lang' => $reviewData['lang'] ?? 'en', // Default to 'en' if not provided
                        'text' => $reviewData['comment'],
                        'status' => 'new',
                        'reviewed_at' => isset($reviewData['createTime']) ?
                            \Carbon\Carbon::parse($reviewData['createTime']) :
                            now(),
                    ]
                );

                if ($review->wasRecentlyCreated) {
                    $newReviews++;
                } else {
                    $updatedReviews++;
                }

                $ingestedCount++;
            }

            Log::info('Successfully ingested reviews', [
                'location_name' => $location->display_name,
                'total_ingested' => $ingestedCount,
                'new_reviews' => $newReviews,
                'updated_reviews' => $updatedReviews
            ]);

            return [
                'success' => true,
                'reviews_ingested' => $ingestedCount,
                'new_reviews' => $newReviews,
                'updated_reviews' => $updatedReviews,
                'location_name' => $location->display_name,
            ];

        } catch (\Exception $e) {
            Log::error('Failed to ingest reviews for location', [
                'location_name' => $location->display_name,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Convert text star ratings to integers
     *
     * @param mixed $stars
     * @return int|null
     */
    private function convertStarRatingToInt($stars)
    {
        // Handle integer values
        if (is_numeric($stars)) {
            $intValue = (int) $stars;
            return ($intValue >= 1 && $intValue <= 5) ? $intValue : null;
        }

        // Handle text values
        if (is_string($stars)) {
            $stars = strtoupper(trim($stars));

            $mapping = [
                'ONE' => 1,
                'TWO' => 2,
                'THREE' => 3,
                'FOUR' => 4,
                'FIVE' => 5,
                '1' => 1,
                '2' => 2,
                '3' => 3,
                '4' => 4,
                '5' => 5,
            ];

            return $mapping[$stars] ?? null;
        }

        return null;
    }
}
