<?php

namespace App\Services;

use App\Models\MailAccount;
use Illuminate\Support\Facades\Log;
use App\Support\SecureHttpClient;

class IMAPEmailService
{
    private $imapConnection = null;
    private $mailAccount = null;

    /**
     * Test IMAP connection without fetching emails
     */
    public function testConnection(MailAccount $mailAccount): array
    {
        try {
            $config = $mailAccount->imap_config;
            
            if (empty($config['host']) || empty($config['port']) || 
                empty($config['username']) || empty($config['password'])) {
                return [
                    'success' => false,
                    'message' => '❌ Incomplete IMAP configuration. Missing host, port, username, or password.'
                ];
            }

            // Build IMAP connection string
            $connectionString = $this->buildConnectionString($config);
            
            Log::info('Testing IMAP connection', [
                'host' => $config['host'],
                'port' => $config['port'],
                'username' => $config['username']
            ]);

            // Suppress PHP warnings that could corrupt JSON response
            $oldErrorReporting = error_reporting(E_ERROR | E_PARSE | E_CORE_ERROR | E_COMPILE_ERROR | E_USER_ERROR | E_RECOVERABLE_ERROR);

            // Increase timeout for IMAP operations
            set_time_limit(60);
            imap_timeout(IMAP_OPENTIMEOUT, 30);
            imap_timeout(IMAP_READTIMEOUT, 30);
            imap_timeout(IMAP_WRITETIMEOUT, 30);
            imap_timeout(IMAP_CLOSETIMEOUT, 10);

            try {
                // Test connection with read-only flag to prevent accidental message deletion
                // CL_EXPUNGE was removed as it automatically deletes messages marked for deletion
                $connection = @imap_open($connectionString, $config['username'], $config['password'], OP_READONLY, 1);
            } finally {
                // Restore original error reporting
                error_reporting($oldErrorReporting);
            }
            
            if (!$connection) {
                $errors = imap_errors();
                $lastError = imap_last_error();
                $errorMessage = !empty($errors) ? implode('; ', $errors) : 'Unknown IMAP error';
                
                // Check if it's an authentication issue
                if (strpos(strtolower($errorMessage), 'authentication') !== false || 
                    strpos(strtolower($errorMessage), 'login') !== false ||
                    strpos(strtolower($errorMessage), 'credentials') !== false) {
                    
                    return [
                        'success' => false,
                        'message' => "❌ IMAP authentication failed: {$errorMessage}\n\n" .
                                   "🔧 Troubleshooting:\n" .
                                   "• SMTP and IMAP passwords may be different\n" .
                                   "• Some providers require app-specific passwords\n" .
                                   "• Check if IMAP is enabled in email settings\n" .
                                   "• Try using your main email password\n" .
                                   "• Contact your email provider for IMAP setup"
                    ];
                }
                
                return [
                    'success' => false,
                    'message' => "❌ IMAP connection failed: {$errorMessage}\n\n" .
                               "🔧 Troubleshooting:\n" .
                               "• Check if IMAP host is correct\n" .
                               "• Try common hosts: imap.hostinger.com, imap.gmail.com\n" .
                               "• Verify port: 993 for SSL, 143 for TLS\n" .
                               "• Check login credentials\n" .
                               "• Ensure IMAP is enabled in your email provider settings"
                ];
            }

            // Test folder access with fallback folder names
            $folderNames = [
                $config['folder'], 
                'INBOX', 
                'Inbox', 
                'inbox', 
                'MAILBOX', 
                'Mailbox'
            ];
            
            $folderInfo = null;
            $workingFolder = null;
            
            foreach ($folderNames as $folderName) {
                $folderInfo = @imap_status($connection, $folderName, SA_MESSAGES | SA_UNSEEN | SA_RECENT);
                if ($folderInfo) {
                    $workingFolder = $folderName;
                    break;
                }
            }
            
            if (!$folderInfo) {
                imap_close($connection);
                
                // Try to list available folders
                $availableFolders = @imap_list($connection, $connectionString, "*");
                $folderList = $availableFolders ? implode(', ', array_map(function($folder) {
                    return preg_replace('/^\{.*?\}+/', '', $folder); // Remove server prefix
                }, $availableFolders)) : 'Unable to retrieve folder list';
                
                return [
                    'success' => false,
                    'message' => "❌ Cannot access inbox folder. Available folders: {$folderList}\n\n" .
                               "🔧 Try using one of these folder names instead of '{$config['folder']}'"
                ];
            }

            imap_close($connection);

            return [
                'success' => true,
                'message' => "✅ IMAP connection successful!\n\n" .
                           "📁 Working folder: {$workingFolder}\n" .
                           "📧 Total messages: {$folderInfo->messages}\n" .
                           "📬 Unread messages: {$folderInfo->unseen}\n" .
                           "📮 Recent messages: {$folderInfo->recent}",
                'recommended_folder' => $workingFolder
            ];

        } catch (\Exception $e) {
            Log::error('IMAP connection test failed', [
                'error' => $e->getMessage(),
                'account' => $mailAccount->email
            ]);

            return [
                'success' => false,
                'message' => '❌ IMAP connection test failed: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Fetch emails from IMAP account
     */
    public function fetchEmails(MailAccount $mailAccount, int $limit = 20): array
    {
        try {
            // Check if IMAP extension is available
            if (!extension_loaded('imap')) {
                return [
                    'success' => false,
                    'message' => '❌ IMAP functionality requires PHP IMAP extension. Please install php-imap.'
                ];
            }

            $config = $mailAccount->imap_config;
            
            if (!$mailAccount->hasImapConfig()) {
                return [
                    'success' => false,
                    'message' => 'IMAP not configured for this account'
                ];
            }

            $connectionString = $this->buildConnectionString($config);
            
            Log::info('Fetching emails from IMAP', [
                'account' => $mailAccount->email,
                'host' => $config['host'],
                'folder' => $config['folder']
            ]);

            // Suppress PHP warnings that could corrupt JSON response
            $oldErrorReporting = error_reporting(E_ERROR | E_PARSE | E_CORE_ERROR | E_COMPILE_ERROR | E_USER_ERROR | E_RECOVERABLE_ERROR);

            // Increase timeout for IMAP operations
            set_time_limit(120);
            imap_timeout(IMAP_OPENTIMEOUT, 30);
            imap_timeout(IMAP_READTIMEOUT, 30);
            imap_timeout(IMAP_WRITETIMEOUT, 30);
            imap_timeout(IMAP_CLOSETIMEOUT, 10);

            try {
                // Use read-only mode to prevent accidental message deletion
                // CL_EXPUNGE was removed as it automatically deletes messages marked for deletion
                $connection = @imap_open($connectionString, $config['username'], $config['password'], OP_READONLY, 1);
            } finally {
                // Restore original error reporting
                error_reporting($oldErrorReporting);
            }

            if (!$connection) {
                $errors = imap_errors();
                $errorMessage = !empty($errors) ? implode('; ', $errors) : 'Unknown IMAP error';

                throw new \Exception("IMAP connection failed: {$errorMessage}");
            }

            // The connection is already opened to the default folder (INBOX)
            // Just search for messages directly
            $searchCriteria = 'ALL';
            $messages = @imap_search($connection, $searchCriteria);
            
            if (empty($messages)) {
                imap_close($connection);
                return [
                    'success' => true,
                    'count' => 0,
                    'message' => 'No emails found',
                    'emails' => []
                ];
            }
            
            // Sort messages by date (newest last)
            rsort($messages);

            // Process latest messages (limit)
            $emailCount = min(count($messages), $limit);
            $processedEmails = [];

            for ($i = 0; $i < $emailCount; $i++) {
                $messageNumber = $messages[$i];
                $header = imap_headerinfo($connection, $messageNumber);
                
                if (!$header) continue;

                // Extract email data
                $emailData = $this->extractEmailData($connection, $header, $messageNumber);
                
                if ($emailData) {
                    $processedEmails[] = $emailData;
                }
            }

            imap_close($connection);

            Log::info('IMAP fetch completed', [
                'account' => $mailAccount->email,
                'fetched_count' => count($processedEmails),
                'total_available' => count($messages)
            ]);

            return [
                'success' => true,
                'count' => count($processedEmails),
                'message' => 'Emails fetched successfully',
                'emails' => $processedEmails
            ];

        } catch (\Exception $e) {
            Log::error('IMAP fetch failed', [
                'error' => $e->getMessage(),
                'account' => $mailAccount->email
            ]);

            return [
                'success' => false,
                'message' => 'Failed to fetch emails: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Build IMAP connection string
     */
    private function buildConnectionString(array $config): string
    {
        $flags = [];
        
        if ($config['encryption'] === 'ssl') {
            $flags[] = 'ssl';
        } elseif ($config['encryption'] === 'tls') {
            $flags[] = 'tls';
        }
        
        // Add SSL validation flags based on environment
        if (SecureHttpClient::shouldDisableSslVerification()) {
            $flags[] = 'novalidate-cert'; // Skip SSL certificate validation in local dev only
        }
        $flags[] = 'norsh'; // Don't use IMAP QRESYNC
        
        $flagsString = !empty($flags) ? '/' . implode('/', $flags) : '';
        
        return "{" . $config['host'] . ":" . $config['port'] . $flagsString . "}";
    }

    /**
     * Check if IMAP and SMTP passwords might be different for this domain
     */
    public function requiresSeparateImapPassword(string $email): bool
    {
        $domain = strtolower(substr(strrchr($email, "@"), 1));
        
        // Domains that typically require separate IMAP passwords
        $separatePasswordDomains = [
            'gmail.com',
            'googlemail.com', 
            'outlook.com',
            'hotmail.com',
            'live.com',
            'yahoo.com',
            'ymail.com',
            'aol.com'
        ];
        
        return in_array($domain, $separatePasswordDomains);
    }

    /**
     * Auto-resolve IMAP host based on email domain
     */
    public function resolveImapHost(string $email): string
    {
        $domain = strtolower(substr(strrchr($email, "@"), 1));
        
        // Common IMAP host patterns
        $imapHosts = [
            'gmail.com' => 'imap.gmail.com',
            'googlemail.com' => 'imap.gmail.com',
            'outlook.com' => 'outlook.office365.com',
            'hotmail.com' => 'outlook.office365.com',
            'live.com' => 'outlook.office365.com',
            'yahoo.com' => 'imap.mail.yahoo.com',
            'ymail.com' => 'imap.mail.yahoo.com',
            'aol.com' => 'imap.aol.com',
        ];
        
        // Provider-specific domains
        $providerPatterns = [
            'hostinger' => 'imap.hostinger.com',
            'godaddy' => 'imap.secureserver.net',
            'bluehost' => 'imap.bluehost.com',
            'siteground' => 'imap.siteground.com',
            'namecheap' => 'mail.privateemail.com',
            'name.com' => 'imap.name.com',
            'titan.email' => 'imap.titan.email',
        ];
        
        // Check exact matches first
        if (isset($imapHosts[$domain])) {
            return $imapHosts[$domain];
        }
        
        // Check provider patterns
        foreach ($providerPatterns as $provider => $host) {
            if (strpos($domain, $provider) !== false) {
                return $host;
            }
        }
        
        // Default patterns for generic hosts
        return "mail.{$domain}"; // Try mail.domain.com
    }

    /**
     * Extract email data from IMAP message
     */
    private function extractEmailData($connection, $header, int $messageNumber): ?array
    {
        try {
            // Get message body
            $structure = imap_fetchstructure($connection, $messageNumber);
            $body = $this->extractMessageBody($connection, $messageNumber, $structure);

            return [
                'provider_msg_id' => $messageNumber, // Use message number as ID for IMAP
                'subject' => $this->decodeHeader($header->subject ?? ''),
                'from_addr' => $this->extractEmailAddress($header->from[0] ?? null),
                'to_json' => $this->extractRecipients($header->to ?? []),
                'cc_json' => $this->extractRecipients($header->cc ?? []),
                'date' => $this->convertDate($header->date ?? ''),
                'snippet' => substr(strip_tags($body['text'] ?? ''), 0, 250),
                'body_text' => $body['text'] ?? '',
                'body_html' => $body['html'] ?? '',
                'attachments' => $body['attachments'] ?? []
            ];

        } catch (\Exception $e) {
            Log::warning('Failed to extract email data', [
                'message_number' => $messageNumber,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Extract message body content
     */
    private function extractMessageBody($connection, int $messageNumber, $structure): array
    {
        $body = ['text' => '', 'html' => '', 'attachments' => []];
        
        if (isset($structure->parts)) {
            foreach ($structure->parts as $partNumber => $part) {
                $subBody = $this->extractPartBody($connection, $messageNumber, $partNumber + 1, $part);
                
                if ($part->subtype === 'PLAIN') {
                    $body['text'] = $subBody['content'];
                } elseif ($part->subtype === 'HTML') {
                    $body['html'] = $subBody['content'];
                } elseif ($subBody['filename']) {
                    $body['attachments'][] = $subBody;
                }
            }
        } else {
            // Single part message
            $subBody = $this->extractPartBody($connection, $messageNumber, 1, $structure);
            
            if ($structure->subtype === 'PLAIN') {
                $body['text'] = $subBody['content'];
            } elseif ($structure->subtype === 'HTML') {
                $body['html'] = $subBody['content'];
            }
        }
        
        return $body;
    }

    /**
     * Extract body content from a specific part
     */
    private function extractPartBody($connection, int $messageNumber, int $partNumber, $part): array
    {
        $content = '';
        $filename = '';
        
        // Get encoding
        $encoding = $part->encoding ?? 1; // Default to binary
        
        // Fetch the part
        $data = imap_fetchbody($connection, $messageNumber, $partNumber);
        
        // Decode based on encoding
        switch ($encoding) {
            case 3: // BASE64
                $content = base64_decode($data);
                break;
            case 4: // QUOTED-PRINTABLE
                $content = imap_qprint($data);
                break;
            case 0: // 7BIT
            case 1: // 8BIT
            case 2: // BINARY
            default:
                $content = $data;
                break;
        }
        
        // Get filename if attachment
        if (isset($part->parameters)) {
            foreach ($part->parameters as $param) {
                if (strtolower($param->attribute) === 'name') {
                    $filename = $this->decodeHeader($param->value);
                    break;
                }
            }
        }
        
        return [
            'content' => $content,
            'filename' => $filename,
            'type' => $part->subtype ?? 'unknown'
        ];
    }

    /**
     * Decode email headers (handle MIME encoding)
     */
    private function decodeHeader(?string $header): string
    {
        if (empty($header)) return '';
        
        return iconv_mime_decode($header, ICONV_MIME_DECODE_CONTINUE_ON_ERROR, 'UTF-8');
    }

    /**
     * Extract email address from header object
     */
    private function extractEmailAddress($address): string
    {
        if (!$address || !isset($address->mailbox) || !isset($address->host)) {
            return '';
        }
        
        return $address->mailbox . '@' . $address->host;
    }

    /**
     * Extract recipients from header array
     */
    private function extractRecipients(array $recipients): array
    {
        $emails = [];
        
        foreach ($recipients as $recipient) {
            $email = $this->extractEmailAddress($recipient);
            if (!empty($email)) {
                $emails[] = $email;
            }
        }
        
        return $emails;
    }

    /**
     * Convert IMAP date to standard format
     */
    private function convertDate(string $date): ?string
    {
        if (empty($date)) return null;
        
        try {
            $dateTime = new \DateTime($date);
            return $dateTime->format('Y-m-d H:i:s');
        } catch (\Exception $e) {
            return null;
        }
    }
}
