<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Cache;

class ErrorTrackingService
{
    /**
     * Track and report errors with context
     */
    public static function trackError(string $type, string $message, array $context = []): void
    {
        $errorData = [
            'type' => $type,
            'message' => $message,
            'context' => $context,
            'timestamp' => now()->toISOString(),
            'user_id' => auth()->id(),
            'url' => request()->fullUrl(),
            'ip' => request()->ip(),
        ];

        // Log the error
        Log::error("Error tracked: {$type}", $errorData);

        // Store in cache for monitoring dashboard
        $errorKey = "error_tracking:" . now()->format('Y-m-d-H');
        $errors = Cache::get($errorKey, []);
        $errors[] = $errorData;
        Cache::put($errorKey, $errors, 3600); // Keep for 1 hour

        // Check if we should send alerts
        self::checkForAlerts($type, $errorData);
    }

    /**
     * Track performance metrics
     */
    public static function trackPerformance(string $operation, float $duration, array $context = []): void
    {
        $perfData = [
            'operation' => $operation,
            'duration' => $duration,
            'context' => $context,
            'timestamp' => now()->toISOString(),
            'memory_usage' => memory_get_usage(true),
            'peak_memory' => memory_get_peak_usage(true),
        ];

        Log::info("Performance tracked: {$operation}", $perfData);

        // Store performance metrics
        $perfKey = "performance:" . now()->format('Y-m-d-H');
        $metrics = Cache::get($perfKey, []);
        $metrics[] = $perfData;
        Cache::put($perfKey, $metrics, 3600);
    }

    /**
     * Track user actions for audit
     */
    public static function trackUserAction(string $action, array $context = []): void
    {
        $actionData = [
            'action' => $action,
            'user_id' => auth()->id(),
            'context' => $context,
            'timestamp' => now()->toISOString(),
            'ip' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ];

        Log::info("User action tracked: {$action}", $actionData);

        // Store in audit log
        $auditKey = "audit_log:" . now()->format('Y-m-d');
        $auditLog = Cache::get($auditKey, []);
        $auditLog[] = $actionData;
        Cache::put($auditKey, $auditLog, 86400); // Keep for 24 hours
    }

    /**
     * Check if we should send alerts based on error frequency
     */
    private static function checkForAlerts(string $type, array $errorData): void
    {
        $alertKey = "error_alerts:{$type}";
        $errorCount = Cache::increment($alertKey, 1);
        Cache::expire($alertKey, 300); // 5 minutes

        // Send alert if error count exceeds threshold
        if ($errorCount >= 5) {
            self::sendErrorAlert($type, $errorCount, $errorData);
            Cache::forget($alertKey); // Reset counter
        }
    }

    /**
     * Send error alert to administrators
     */
    private static function sendErrorAlert(string $type, int $count, array $errorData): void
    {
        try {
            // This would typically send to admin email or notification service
            Log::critical("Error alert: {$type} occurred {$count} times in 5 minutes", $errorData);
            
            // Example: Send email to admin
            // Mail::to(config('app.admin_email'))->send(new ErrorAlertMail($type, $count, $errorData));
        } catch (\Exception $e) {
            Log::error('Failed to send error alert', ['error' => $e->getMessage()]);
        }
    }

    /**
     * Get error statistics
     */
    public static function getErrorStats(string $date = null): array
    {
        $date = $date ?: now()->format('Y-m-d');
        $errors = [];
        
        // Collect errors from cache for the day
        for ($hour = 0; $hour < 24; $hour++) {
            $key = "error_tracking:{$date}-" . str_pad($hour, 2, '0', STR_PAD_LEFT);
            $hourErrors = Cache::get($key, []);
            $errors = array_merge($errors, $hourErrors);
        }

        return [
            'total_errors' => count($errors),
            'error_types' => array_count_values(array_column($errors, 'type')),
            'hourly_distribution' => self::getHourlyDistribution($errors),
            'top_errors' => self::getTopErrors($errors),
        ];
    }

    /**
     * Get performance statistics
     */
    public static function getPerformanceStats(string $date = null): array
    {
        $date = $date ?: now()->format('Y-m-d');
        $metrics = [];
        
        for ($hour = 0; $hour < 24; $hour++) {
            $key = "performance:{$date}-" . str_pad($hour, 2, '0', STR_PAD_LEFT);
            $hourMetrics = Cache::get($key, []);
            $metrics = array_merge($metrics, $hourMetrics);
        }

        return [
            'total_operations' => count($metrics),
            'avg_duration' => array_sum(array_column($metrics, 'duration')) / count($metrics),
            'slowest_operations' => self::getSlowestOperations($metrics),
            'memory_usage' => self::getMemoryStats($metrics),
        ];
    }

    private static function getHourlyDistribution(array $errors): array
    {
        $distribution = [];
        foreach ($errors as $error) {
            $hour = date('H', strtotime($error['timestamp']));
            $distribution[$hour] = ($distribution[$hour] ?? 0) + 1;
        }
        return $distribution;
    }

    private static function getTopErrors(array $errors): array
    {
        $errorMessages = array_column($errors, 'message');
        return array_count_values($errorMessages);
    }

    private static function getSlowestOperations(array $metrics): array
    {
        usort($metrics, function($a, $b) {
            return $b['duration'] <=> $a['duration'];
        });
        return array_slice($metrics, 0, 10);
    }

    private static function getMemoryStats(array $metrics): array
    {
        return [
            'avg_memory' => array_sum(array_column($metrics, 'memory_usage')) / count($metrics),
            'peak_memory' => max(array_column($metrics, 'peak_memory')),
        ];
    }
}
