<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Auth;

class AuditLogService
{
    const AUDIT_PREFIX = 'audit:';
    const RETENTION_DAYS = 90;

    /**
     * Log user authentication events
     */
    public static function logAuthEvent(string $event, array $context = []): void
    {
        $auditData = [
            'event_type' => 'authentication',
            'event' => $event,
            'user_id' => Auth::id(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'timestamp' => now()->toISOString(),
            'context' => $context,
        ];

        self::storeAuditLog($auditData);
        Log::info("Audit: Auth event - {$event}", $auditData);
    }

    /**
     * Log email operations
     */
    public static function logEmailOperation(string $operation, array $context = []): void
    {
        $auditData = [
            'event_type' => 'email_operation',
            'event' => $operation,
            'user_id' => Auth::id(),
            'ip_address' => request()->ip(),
            'timestamp' => now()->toISOString(),
            'context' => $context,
        ];

        self::storeAuditLog($auditData);
        Log::info("Audit: Email operation - {$operation}", $auditData);
    }

    /**
     * Log account management events
     */
    public static function logAccountEvent(string $event, array $context = []): void
    {
        $auditData = [
            'event_type' => 'account_management',
            'event' => $event,
            'user_id' => Auth::id(),
            'ip_address' => request()->ip(),
            'timestamp' => now()->toISOString(),
            'context' => $context,
        ];

        self::storeAuditLog($auditData);
        Log::info("Audit: Account event - {$event}", $auditData);
    }

    /**
     * Log security events
     */
    public static function logSecurityEvent(string $event, array $context = []): void
    {
        $auditData = [
            'event_type' => 'security',
            'event' => $event,
            'user_id' => Auth::id(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'timestamp' => now()->toISOString(),
            'context' => $context,
            'severity' => $context['severity'] ?? 'medium',
        ];

        self::storeAuditLog($auditData);
        Log::warning("Audit: Security event - {$event}", $auditData);
    }

    /**
     * Log API access
     */
    public static function logApiAccess(string $endpoint, array $context = []): void
    {
        $auditData = [
            'event_type' => 'api_access',
            'event' => 'api_request',
            'endpoint' => $endpoint,
            'user_id' => Auth::id(),
            'ip_address' => request()->ip(),
            'timestamp' => now()->toISOString(),
            'context' => $context,
        ];

        self::storeAuditLog($auditData);
    }

    /**
     * Store audit log entry
     */
    private static function storeAuditLog(array $auditData): void
    {
        $date = now()->format('Y-m-d');
        $key = self::AUDIT_PREFIX . $date;
        
        $logs = Cache::get($key, []);
        $logs[] = $auditData;
        
        // Store for 24 hours initially
        Cache::put($key, $logs, 86400);
        
        // Also store in daily file for long-term retention
        self::storeToFile($auditData);
    }

    /**
     * Store audit log to file for long-term retention
     */
    private static function storeToFile(array $auditData): void
    {
        $date = now()->format('Y-m-d');
        $logFile = storage_path("logs/audit-{$date}.log");
        
        $logEntry = json_encode($auditData) . "\n";
        file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    }

    /**
     * Get audit logs for a specific date range
     */
    public static function getAuditLogs(string $startDate, string $endDate, array $filters = []): array
    {
        $logs = [];
        $currentDate = $startDate;
        
        while ($currentDate <= $endDate) {
            $key = self::AUDIT_PREFIX . $currentDate;
            $dayLogs = Cache::get($key, []);
            $logs = array_merge($logs, $dayLogs);
            $currentDate = date('Y-m-d', strtotime($currentDate . ' +1 day'));
        }

        // Apply filters
        if (!empty($filters)) {
            $logs = array_filter($logs, function($log) use ($filters) {
                foreach ($filters as $key => $value) {
                    if (isset($log[$key]) && $log[$key] !== $value) {
                        return false;
                    }
                }
                return true;
            });
        }

        return $logs;
    }

    /**
     * Get security events
     */
    public static function getSecurityEvents(int $days = 7): array
    {
        $startDate = now()->subDays($days)->format('Y-m-d');
        $endDate = now()->format('Y-m-d');
        
        return self::getAuditLogs($startDate, $endDate, ['event_type' => 'security']);
    }

    /**
     * Get user activity summary
     */
    public static function getUserActivitySummary(int $userId, int $days = 30): array
    {
        $startDate = now()->subDays($days)->format('Y-m-d');
        $endDate = now()->format('Y-m-d');
        
        $logs = self::getAuditLogs($startDate, $endDate, ['user_id' => $userId]);
        
        $summary = [
            'total_events' => count($logs),
            'event_types' => array_count_values(array_column($logs, 'event_type')),
            'daily_activity' => [],
            'ip_addresses' => array_unique(array_column($logs, 'ip_address')),
            'recent_events' => array_slice($logs, -10),
        ];

        // Group by day
        foreach ($logs as $log) {
            $day = date('Y-m-d', strtotime($log['timestamp']));
            $summary['daily_activity'][$day] = ($summary['daily_activity'][$day] ?? 0) + 1;
        }

        return $summary;
    }

    /**
     * Detect suspicious activity
     */
    public static function detectSuspiciousActivity(): array
    {
        $suspiciousEvents = [];
        
        // Check for multiple failed login attempts
        $failedLogins = self::getAuditLogs(
            now()->subHours(1)->format('Y-m-d'),
            now()->format('Y-m-d'),
            ['event' => 'login_failed']
        );
        
        $loginAttemptsByIp = [];
        foreach ($failedLogins as $login) {
            $ip = $login['ip_address'];
            $loginAttemptsByIp[$ip] = ($loginAttemptsByIp[$ip] ?? 0) + 1;
        }
        
        foreach ($loginAttemptsByIp as $ip => $count) {
            if ($count >= 5) {
                $suspiciousEvents[] = [
                    'type' => 'multiple_failed_logins',
                    'ip_address' => $ip,
                    'count' => $count,
                    'severity' => 'high'
                ];
            }
        }

        // Check for unusual API access patterns
        $apiAccess = self::getAuditLogs(
            now()->subHours(1)->format('Y-m-d'),
            now()->format('Y-m-d'),
            ['event_type' => 'api_access']
        );
        
        $apiAccessByUser = [];
        foreach ($apiAccess as $access) {
            $userId = $access['user_id'];
            $apiAccessByUser[$userId] = ($apiAccessByUser[$userId] ?? 0) + 1;
        }
        
        foreach ($apiAccessByUser as $userId => $count) {
            if ($count >= 100) {
                $suspiciousEvents[] = [
                    'type' => 'excessive_api_access',
                    'user_id' => $userId,
                    'count' => $count,
                    'severity' => 'medium'
                ];
            }
        }

        return $suspiciousEvents;
    }

    /**
     * Clean up old audit logs
     */
    public static function cleanupOldLogs(): int
    {
        $cutoffDate = now()->subDays(self::RETENTION_DAYS)->format('Y-m-d');
        $currentDate = now()->subDays(self::RETENTION_DAYS + 1)->format('Y-m-d');
        $deletedCount = 0;
        
        // Clean up cache entries
        while ($currentDate <= $cutoffDate) {
            $key = self::AUDIT_PREFIX . $currentDate;
            if (Cache::has($key)) {
                Cache::forget($key);
                $deletedCount++;
            }
            $currentDate = date('Y-m-d', strtotime($currentDate . ' +1 day'));
        }
        
        // Clean up log files
        $logFiles = glob(storage_path('logs/audit-*.log'));
        foreach ($logFiles as $file) {
            $fileDate = basename($file, '.log');
            $fileDate = str_replace('audit-', '', $fileDate);
            if ($fileDate < $cutoffDate) {
                unlink($file);
                $deletedCount++;
            }
        }
        
        return $deletedCount;
    }
}
