<?php

namespace App\Services\AIProviders;

use App\Contracts\AIProviderInterface;
use App\Support\RetryHelper;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class AnthropicProvider implements AIProviderInterface
{
    private ?string $apiKey;
    private string $model;

    public function __construct()
    {
        $this->apiKey = config('services.anthropic.api_key');
        $this->model = config('services.anthropic.model', 'claude-3-haiku-20240307');
    }

    public function generateResponse(string $systemPrompt, string $userPrompt, array $options = []): array
    {
        try {
            if (!$this->apiKey) {
                throw new \Exception('Anthropic API key not configured');
            }

            // Use retry helper with exponential backoff for transient failures
            $response = RetryHelper::retry(
                fn() => Http::timeout(30)->withHeaders([
                    'x-api-key' => $this->apiKey,
                    'anthropic-version' => '2023-06-01',
                    'content-type' => 'application/json',
                ])->post('https://api.anthropic.com/v1/messages', [
                    'model' => $this->model,
                    'max_tokens' => $options['max_tokens'] ?? 200,
                    'temperature' => $options['temperature'] ?? 0.7,
                    'system' => $systemPrompt,
                    'messages' => [
                        [
                            'role' => 'user',
                            'content' => $userPrompt,
                        ],
                    ],
                ]),
                maxAttempts: 3,
                baseDelayMs: 1000
            );

            if (!$response->successful()) {
                throw new \Exception('Anthropic API request failed: ' . $response->body());
            }

            $data = $response->json();
            $content = $data['content'][0]['text'] ?? '';

            return [
                'success' => true,
                'content' => trim($content),
                'provider' => 'anthropic',
            ];
        } catch (\Exception $e) {
            Log::error('Anthropic API error after retries: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'provider' => 'anthropic',
            ];
        }
    }

    public function testConnection(): array
    {
        try {
            if (!$this->apiKey) {
                throw new \Exception('Anthropic API key not configured');
            }

            // Include system prompt to match the actual API call structure
            $response = Http::timeout(30)->withHeaders([
                'x-api-key' => $this->apiKey,
                'anthropic-version' => '2023-06-01',
                'content-type' => 'application/json',
            ])->post('https://api.anthropic.com/v1/messages', [
                'model' => $this->model,
                'max_tokens' => 20,
                'system' => 'You are a helpful assistant performing a connection test.',
                'messages' => [
                    [
                        'role' => 'user',
                        'content' => 'Say "Anthropic connection test successful"',
                    ],
                ],
            ]);

            if (!$response->successful()) {
                throw new \Exception('Anthropic API request failed: ' . $response->body());
            }

            $data = $response->json();
            $content = $data['content'][0]['text'] ?? '';

            return [
                'success' => true,
                'message' => 'Anthropic connection successful',
                'response' => trim($content),
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function getProviderName(): string
    {
        return 'Claude';
    }

    public function isConfigured(): bool
    {
        return !empty($this->apiKey);
    }
}
