<?php

namespace App\Modules\SMTP\Services;

use App\Models\MailAccount;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class SMTPService
{
    /**
     * Send email via SMTP
     */
    public function sendEmail(MailAccount $account, array $emailData): bool
    {
        try {
            $config = [
                'driver' => 'smtp',
                'host' => $account->smtp_host,
                'port' => $account->smtp_port,
                'username' => $account->smtp_username,
                'password' => $account->smtp_password,
                'encryption' => $account->smtp_encryption,
                'from' => [
                    'address' => $account->email,
                    'name' => $account->from_name,
                ],
            ];

            config(['mail.mailers.smtp' => $config]);

            Mail::raw($emailData['body'], function ($message) use ($emailData, $account) {
                $message->to($emailData['to'])
                        ->subject($emailData['subject'])
                        ->from($account->email, $account->from_name);
            });

            return true;
        } catch (\Exception $e) {
            Log::error('SMTP send error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Test SMTP connection
     */
    public function testConnection(MailAccount $account): array
    {
        try {
            $config = [
                'driver' => 'smtp',
                'host' => $account->smtp_host,
                'port' => $account->smtp_port,
                'username' => $account->smtp_username,
                'password' => $account->smtp_password,
                'encryption' => $account->smtp_encryption,
            ];

            config(['mail.mailers.smtp' => $config]);

            // Test connection by sending a test email
            Mail::raw('Test email from ReplyPilot', function ($message) use ($account) {
                $message->to($account->email)
                        ->subject('SMTP Connection Test')
                        ->from($account->email, $account->from_name);
            });

            return [
                'success' => true,
                'message' => 'SMTP connection successful'
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'SMTP connection failed: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Fetch emails via IMAP
     */
    public function fetchEmails(MailAccount $account, int $limit = 50): array
    {
        try {
            $connectionString = "{{$account->imap_host}:{$account->imap_port}/{$account->imap_encryption}/novalidate-cert}INBOX";
            
            $inbox = imap_open($connectionString, $account->imap_username, $account->imap_password);
            
            if (!$inbox) {
                throw new \Exception('IMAP connection failed: ' . imap_last_error());
            }

            $emails = [];
            $messageCount = imap_num_msg($inbox);
            $start = max(1, $messageCount - $limit + 1);

            for ($i = $start; $i <= $messageCount; $i++) {
                $header = imap_headerinfo($inbox, $i);
                $body = imap_body($inbox, $i);
                
                $emails[] = [
                    'id' => $i,
                    'subject' => $header->subject ?? '',
                    'from' => $header->from[0]->mailbox . '@' . $header->from[0]->host,
                    'date' => $header->date ?? '',
                    'body' => $body,
                ];
            }

            imap_close($inbox);
            return $emails;
        } catch (\Exception $e) {
            Log::error('IMAP fetch error: ' . $e->getMessage());
            return [];
        }
    }
}
