<?php

namespace App\Modules\Reviews\Services;

use App\Models\Review;
use App\Models\ReviewDraft;
use App\Models\Location;
use App\Services\GoogleBusinessProfile;
use App\Services\EnhancedReplyDraftService;
use App\Services\AIServiceManager;
use Illuminate\Support\Facades\Log;

class ReviewsService
{
    private GoogleBusinessProfile $gbpService;
    private EnhancedReplyDraftService $draftService;

    public function __construct(GoogleBusinessProfile $gbpService, AIServiceManager $aiManager)
    {
        $this->gbpService = $gbpService;
        $this->draftService = new EnhancedReplyDraftService($aiManager);
    }

    /**
     * Import Google Business Profile locations
     */
    public function importLocations(int $userId, int $tenantId): int
    {
        try {
            $gbpLocations = $this->gbpService->listLocations();
            $importedCount = 0;
            
            foreach ($gbpLocations as $gbpLocation) {
                Location::updateOrCreate(
                    [
                        'user_id' => $userId,
                        'tenant_id' => $tenantId,
                        'account_id' => $gbpLocation['account_id'],
                        'location_id' => $gbpLocation['location_id'],
                    ],
                    [
                        'source' => 'google_gbp',
                        'display_name' => $gbpLocation['display_name'],
                    ]
                );
                $importedCount++;
            }
            
            return $importedCount;
        } catch (\Exception $e) {
            Log::error('Failed to import locations: ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Fetch reviews from Google Business Profile
     */
    public function fetchReviews(int $userId, int $tenantId): int
    {
        try {
            $locations = Location::where('user_id', $userId)
                                ->where('tenant_id', $tenantId)
                                ->get();

            if ($locations->count() === 0) {
                return 0;
            }

            $ingestedCount = 0;

            foreach ($locations as $location) {
                $reviews = $this->gbpService->listReviews($location->account_id, $location->location_id);

                foreach ($reviews as $reviewData) {
                    // Convert star rating from text (ONE, TWO, etc.) to integer
                    $starRating = $this->convertStarRatingToInt($reviewData['stars']);
                    if ($starRating === null) {
                        Log::warning('Skipping review with invalid star rating', [
                            'review_id' => $reviewData['id'],
                            'stars' => $reviewData['stars']
                        ]);
                        continue;
                    }

                    Review::updateOrCreate(
                        [
                            'location_id' => $location->id,
                            'external_review_id' => $reviewData['id'],
                        ],
                        [
                            'user_id' => $userId,
                            'tenant_id' => $tenantId,
                            'source' => 'google_gbp',
                            'stars' => $starRating,
                            'lang' => $reviewData['lang'],
                            'text' => $reviewData['comment'],
                            'status' => 'new',
                        ]
                    );
                    $ingestedCount++;
                }
            }

            return $ingestedCount;
        } catch (\Exception $e) {
            Log::error('Failed to fetch reviews: ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Generate AI draft for a review
     */
    public function generateDraft(Review $review): array
    {
        try {
            $result = $this->draftService->makeForReview($review);

            if (!empty($result['meta']['error'])) {
                return [
                    'success' => false,
                    'message' => 'AI draft failed: ' . ($result['meta']['message'] ?? 'Provider error'),
                ];
            }
            
            $draft = ReviewDraft::create([
                'review_id' => $review->id,
                'user_id' => $review->user_id,
                'tenant_id' => $review->tenant_id,
                'draft_reply' => $result['draft'],
                'ai_meta' => $result['meta'],
                'status' => 'pending',
            ]);

            $review->update(['status' => 'drafted']);

            return [
                'success' => true,
                'draft' => $draft,
                'message' => 'AI draft generated successfully'
            ];
        } catch (\Exception $e) {
            Log::error('Failed to generate review draft: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to generate draft: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Approve and post review reply
     */
    public function approveAndPost(Review $review): array
    {
        try {
            $draft = $review->latestDraft();

            if (!$draft) {
                return [
                    'success' => false,
                    'message' => 'No draft found for this review'
                ];
            }

            $result = $this->gbpService->replyToReview(
                $review->location->account_id,
                $review->location->location_id,
                $review->external_review_id,
                $draft->draft_reply
            );

            if ($result['success']) {
                $draft->update(['status' => 'sent']);
                $review->update([
                    'status' => 'posted',
                    'reviewed_at' => now(),
                ]);

                return [
                    'success' => true,
                    'message' => 'Reply posted successfully'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Failed to post reply: ' . $result['message']
                ];
            }
        } catch (\Exception $e) {
            Log::error('Failed to post review reply: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to post reply: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Convert text star ratings to integers
     *
     * @param mixed $stars
     * @return int|null
     */
    private function convertStarRatingToInt($stars)
    {
        // Handle integer values
        if (is_numeric($stars)) {
            $intValue = (int) $stars;
            return ($intValue >= 1 && $intValue <= 5) ? $intValue : null;
        }

        // Handle text values (Google API returns ONE, TWO, THREE, FOUR, FIVE)
        if (is_string($stars)) {
            $stars = strtoupper(trim($stars));

            $mapping = [
                'ONE' => 1,
                'TWO' => 2,
                'THREE' => 3,
                'FOUR' => 4,
                'FIVE' => 5,
                '1' => 1,
                '2' => 2,
                '3' => 3,
                '4' => 4,
                '5' => 5,
            ];

            return $mapping[$stars] ?? null;
        }

        return null;
    }
}
