<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class EmailProvider extends Model
{
    protected $table = 'email_providers';
    
    protected $fillable = [
        'id',
        'name',
        'description',
        'smtp_settings',
        'imap_settings',
        'support_url',
        'logo_url',
        'is_active',
    ];

    protected $casts = [
        'smtp_settings' => 'array',
        'imap_settings' => 'array',
        'is_active' => 'boolean',
    ];

    protected $keyType = 'string';
    public $incrementing = false;

    /**
     * Get SMTP host from settings
     */
    public function getSmtpHostAttribute(): ?string
    {
        return $this->smtp_settings['host'] ?? null;
    }

    /**
     * Get SMTP port from settings
     */
    public function getSmtpPortAttribute(): ?int
    {
        return $this->smtp_settings['port'] ?? null;
    }

    /**
     * Get SMTP encryption from settings
     */
    public function getSmtpEncryptionAttribute(): ?string
    {
        return $this->smtp_settings['encryption'] ?? null;
    }

    /**
     * Check if provider requires app password
     */
    public function requiresAppPassword(): bool
    {
        return $this->smtp_settings['requires_app_password'] ?? false;
    }

    /**
     * Get provider placeholder instructions
     */
    public function getInstructionsAttribute(): string
    {
        $instructions = [];

        if ($this->id === 'hostinger') {
            $instructions[] = 'Replace "yourdomain.com" with your actual domain';
            $instructions[] = 'Use your domain email address (e.g., hello@yourdomain.com)';
        } elseif ($this->id === 'custom') {
            $instructions[] = 'Configure your custom SMTP server settings';
            $instructions[] = 'Ensure your SMTP server allows authentication';
        } else {
            $instructions[] = 'Enter your email credentials';
            $instructions[] = 'Use app password if 2FA is enabled';
        }

        return implode('. ', $instructions) . '.';
    }

    /**
     * Scope for active providers
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope for SMTP providers only
     */
    public function scopeSmtp($query)
    {
        return $query->where('id', '!=', 'oauth');
    }
}