<?php

namespace App\Jobs;

use App\Models\Location;
use App\Services\GoogleBusinessProfile;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ProcessReviewIngestion implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $timeout = 180; // 3 minutes
    public $tries = 3;
    public $backoff = [60, 120, 180];

    protected $location;

    /**
     * Create a new job instance.
     */
    public function __construct(Location $location)
    {
        $this->location = $location;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        Log::info('Starting review ingestion job', [
            'location_id' => $this->location->id,
            'display_name' => $this->location->display_name
        ]);

        try {
            $googleService = app(GoogleBusinessProfile::class);
            $googleService->fetchReviews($this->location);

            Log::info('Review ingestion job completed successfully', [
                'location_id' => $this->location->id
            ]);

        } catch (\Exception $e) {
            Log::error('Review ingestion job failed', [
                'location_id' => $this->location->id,
                'error' => $e->getMessage(),
                'attempt' => $this->attempts()
            ]);

            throw $e;
        }
    }

    /**
     * Handle a job failure.
     */
    public function failed(\Throwable $exception): void
    {
        Log::error('Review ingestion job permanently failed', [
            'business_profile_id' => $this->businessProfile->id,
            'error' => $exception->getMessage(),
            'attempts' => $this->attempts()
        ]);
    }
}
