<?php

namespace App\Jobs;

use App\Models\Location;
use App\Services\ReviewIngestionService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

/**
 * Ingest Reviews Job
 *
 * Background job to fetch reviews for a single Google Business Profile location.
 * Dispatched by IngestAllReviewsCommand, runs in the background via queue worker.
 *
 * Reference: AUTO_INGESTION.md Section 3.2
 */
class IngestReviewsJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * The location to ingest reviews for
     *
     * @var Location
     */
    public $location;

    /**
     * The number of times the job may be attempted.
     *
     * @var int
     */
    public $tries = 3;

    /**
     * The number of seconds the job can run before timing out.
     *
     * @var int
     */
    public $timeout = 300; // 5 minutes

    /**
     * Create a new job instance.
     *
     * @param Location $location
     */
    public function __construct(Location $location)
    {
        $this->location = $location;
    }

    /**
     * Execute the job.
     *
     * @param ReviewIngestionService $ingestionService
     * @return void
     */
    public function handle(ReviewIngestionService $ingestionService)
    {
        Log::info('IngestReviewsJob: Starting review ingestion', [
            'location_name' => $this->location->display_name,
            'location_id' => $this->location->id,
            'attempt' => $this->attempts()
        ]);

        try {
            $result = $ingestionService->ingestForLocation($this->location);

            Log::info('IngestReviewsJob: Successfully completed', [
                'location_name' => $this->location->display_name,
                'reviews_ingested' => $result['reviews_ingested'],
                'new_reviews' => $result['new_reviews'],
                'updated_reviews' => $result['updated_reviews']
            ]);

        } catch (\Exception $e) {
            Log::error('IngestReviewsJob: Failed to ingest reviews', [
                'location_name' => $this->location->display_name,
                'error' => $e->getMessage(),
                'attempt' => $this->attempts()
            ]);

            // Re-throw exception so job will retry
            throw $e;
        }
    }

    /**
     * Handle a job failure.
     *
     * @param \Throwable $exception
     * @return void
     */
    public function failed(\Throwable $exception)
    {
        Log::error('IngestReviewsJob: Job failed after all retries', [
            'location_name' => $this->location->display_name,
            'error' => $exception->getMessage(),
            'attempts' => $this->tries
        ]);

        // Optionally: Send notification to user about failed ingestion
        // Mail::to($this->location->user->email)->send(new ReviewIngestionFailedNotification($this->location));
    }
}
