<?php

namespace App\Jobs;

use App\Models\MailAccount;
use App\Services\EmailIngestionService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

/**
 * Ingest Emails Job
 *
 * Background job to fetch emails for a single mail account.
 * Dispatched by IngestAllEmailsCommand, runs in the background via queue worker.
 *
 * Reference: AUTO_INGESTION.md Section 3.2
 */
class IngestEmailsJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * The mail account to ingest emails for
     *
     * @var MailAccount
     */
    public $mailAccount;

    /**
     * The number of times the job may be attempted.
     *
     * @var int
     */
    public $tries = 3;

    /**
     * The number of seconds the job can run before timing out.
     *
     * @var int
     */
    public $timeout = 300; // 5 minutes

    /**
     * Create a new job instance.
     *
     * @param MailAccount $mailAccount
     */
    public function __construct(MailAccount $mailAccount)
    {
        $this->mailAccount = $mailAccount;
    }

    /**
     * Execute the job.
     *
     * @param EmailIngestionService $ingestionService
     * @return void
     */
    public function handle(EmailIngestionService $ingestionService)
    {
        Log::info('IngestEmailsJob: Starting email ingestion', [
            'account_email' => $this->mailAccount->email,
            'account_id' => $this->mailAccount->id,
            'attempt' => $this->attempts()
        ]);

        try {
            $result = $ingestionService->ingestForAccount($this->mailAccount);

            Log::info('IngestEmailsJob: Successfully completed', [
                'account_email' => $this->mailAccount->email,
                'threads_created' => $result['threads_created'],
                'messages_fetched' => $result['messages_fetched'],
                'messages_created' => $result['messages_created']
            ]);

        } catch (\Exception $e) {
            Log::error('IngestEmailsJob: Failed to ingest emails', [
                'account_email' => $this->mailAccount->email,
                'error' => $e->getMessage(),
                'attempt' => $this->attempts()
            ]);

            // Re-throw exception so job will retry
            throw $e;
        }
    }

    /**
     * Handle a job failure.
     *
     * @param \Throwable $exception
     * @return void
     */
    public function failed(\Throwable $exception)
    {
        Log::error('IngestEmailsJob: Job failed after all retries', [
            'account_email' => $this->mailAccount->email,
            'error' => $exception->getMessage(),
            'attempts' => $this->tries
        ]);

        // Optionally: Send notification to user about failed ingestion
        // Mail::to($this->mailAccount->user->email)->send(new IngestionFailedNotification($this->mailAccount));
    }
}
