<?php

namespace App\Http\Middleware;

use App\Services\SecurityService;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class SecurityValidation
{
    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next)
    {
        // Validate all input for security threats
        $this->validateInput($request);
        
        // Add security headers
        $response = $next($request);
        $this->addSecurityHeaders($response);
        
        return $response;
    }

    /**
     * Validate request input for security threats
     */
    private function validateInput(Request $request): void
    {
        $allInput = $request->all();
        
        foreach ($allInput as $key => $value) {
            if (is_string($value)) {
                // Check for SQL injection
                if (SecurityService::detectSqlInjection($value)) {
                    Log::warning('SQL injection attempt blocked', [
                        'field' => $key,
                        'value' => substr($value, 0, 100),
                        'ip' => $request->ip(),
                        'user_id' => auth()->id(),
                    ]);
                    
                    abort(400, 'Invalid input detected');
                }
                
                // Check for XSS
                if (SecurityService::detectXss($value)) {
                    Log::warning('XSS attempt blocked', [
                        'field' => $key,
                        'value' => substr($value, 0, 100),
                        'ip' => $request->ip(),
                        'user_id' => auth()->id(),
                    ]);
                    
                    abort(400, 'Invalid input detected');
                }
            }
        }
    }

    /**
     * Add security headers to response
     */
    private function addSecurityHeaders($response): void
    {
        $response->headers->set('X-Content-Type-Options', 'nosniff');
        $response->headers->set('X-Frame-Options', 'DENY');
        $response->headers->set('X-XSS-Protection', '1; mode=block');
        $response->headers->set('Referrer-Policy', 'strict-origin-when-cross-origin');
        $response->headers->set('Permissions-Policy', 'geolocation=(), microphone=(), camera=()');
        
        // Content Security Policy
        $csp = "default-src 'self'; " .
               "script-src 'self' 'unsafe-inline'; " .
               "style-src 'self' 'unsafe-inline' https://fonts.googleapis.com; " .
               "font-src 'self' https://fonts.gstatic.com; " .
               "img-src 'self' data: https:; " .
               "connect-src 'self' https://api.openai.com https://generativelanguage.googleapis.com https://api.anthropic.com; " .
               "frame-src 'none'; " .
               "object-src 'none'; " .
               "base-uri 'self'; " .
               "form-action 'self';";
        
        $response->headers->set('Content-Security-Policy', $csp);
        
        // HSTS (only for HTTPS)
        if ($response->getRequest()->secure()) {
            $response->headers->set('Strict-Transport-Security', 'max-age=31536000; includeSubDomains; preload');
        }
    }
}
