<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\EmailDraft;
use App\Models\EmailSignature;
use App\Models\MailAccount;
use App\Models\Message;
use App\Models\Thread;
use App\Services\GmailService;
use App\Services\EnhancedReplyDraftService;
use App\Services\AIServiceManager;
use App\Services\SMTPEmailService;
use App\Services\IMAPEmailService;
use App\Services\UserContextService;
use App\Models\EmailProvider;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class EmailController extends Controller
{
    use AuthorizesRequests;

    private GmailService $gmailService;
    private EnhancedReplyDraftService $draftService;
    private SMTPEmailService $smtpService;
    private IMAPEmailService $imapService;
    private UserContextService $userContext;

    public function __construct(GmailService $gmailService, AIServiceManager $aiManager, SMTPEmailService $smtpService, IMAPEmailService $imapService, UserContextService $userContext)
    {
        $this->gmailService = $gmailService;
        $this->draftService = new EnhancedReplyDraftService($aiManager);
        $this->smtpService = $smtpService;
        $this->imapService = $imapService;
        $this->userContext = $userContext;
    }

    /**
     * Display email threads - Super Admin sees only their own threads
     */
    public function index(Request $request)
    {
        $userId = auth()->id();
        // Super Admin inbox is scoped to their own threads
        $query = Thread::where('user_id', $userId)->with([
            'user' => function ($query) {
                $query->select('id', 'name', 'email');
            },
            'tenant' => function ($query) {
                $query->select('id', 'name');
            },
            'mailAccount' => function ($query) {
                $query->select('id', 'email', 'connection_type');
            },
            'messages' => function ($query) {
                $query->select('id', 'thread_id', 'direction', 'from_addr', 'snippet', 'date')
                      ->orderBy('date', 'desc')
                      ->limit(5);
            }
        ]);
        
        // Filter by mail account if specified
        if ($request->has('account') && $request->account !== 'all') {
            $query = $query->where('mail_account_id', $request->account);
        }
        
        $threads = $query
            ->select('id', 'user_id', 'tenant_id', 'mail_account_id', 'subject', 'participants', 'last_message_at', 'status')
            ->orderBy('last_message_at', 'desc')
            ->paginate(20);
        
        // Super Admin inbox dropdown should list only their own accounts
        $mailAccounts = MailAccount::where('user_id', $userId)
            ->select('id', 'user_id', 'tenant_id', 'email', 'connection_type')
            ->get();
        
        return view('super-admin.email.index', compact('threads', 'mailAccounts'));
    }

    /**
     * Bulk delete selected threads - Super Admin can delete any threads
     */
    public function bulkDelete(Request $request)
    {
        try {
            $threadIds = $request->input('thread_ids', []);
            
            if (empty($threadIds)) {
                return redirect()->route('super-admin.email.index')
                    ->with('error', 'No threads selected for deletion.');
            }

            // Super Admin can delete any threads
            $threadsToDelete = Thread::whereIn('id', $threadIds)->get();

            if ($threadsToDelete->count() !== count($threadIds)) {
                return redirect()->route('super-admin.email.index')
                    ->with('error', 'Some selected threads could not be found.');
            }

            $deletedCount = 0;
            foreach ($threadsToDelete as $thread) {
                // Delete associated messages first
                $thread->messages()->delete();
                
                // Delete associated drafts
                $thread->emailDrafts()->delete();
                
                // Delete the thread
                $thread->delete();
                $deletedCount++;
            }

            return redirect()->route('super-admin.email.index')
                ->with('success', "Successfully deleted {$deletedCount} email thread(s).");

        } catch (\Exception $e) {
            Log::error('Failed to bulk delete threads: ' . $e->getMessage());
            return redirect()->route('super-admin.email.index')
                ->with('error', 'Failed to delete threads: ' . $e->getMessage());
        }
    }

    /**
     * Delete individual thread - Super Admin can delete any thread
     */
    public function destroyThread(Thread $thread)
    {
        try {
            // Delete associated messages first
            $thread->messages()->delete();
            
            // Delete associated drafts
            $thread->emailDrafts()->delete();
            
            // Delete the thread
            $thread->delete();

            return redirect()->route('super-admin.email.index')
                ->with('success', 'Email thread deleted successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to delete thread: ' . $e->getMessage());
            return redirect()->route('super-admin.email.index')
                ->with('error', 'Failed to delete thread: ' . $e->getMessage());
        }
    }
}
