<?php

namespace App\Http\Controllers\Agent;

use App\Http\Controllers\Controller;
use App\Models\Location;
use App\Models\Review;
use App\Models\ReviewDraft;
use App\Models\OauthConnection;
use App\Services\GoogleBusinessProfile;
use App\Services\EnhancedReplyDraftService;
use App\Services\AIServiceManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ReviewController extends Controller
{
    /**
     * Display reviews - Agent sees only their own reviews
     */
    public function index()
    {
        try {
            // Agent sees only their own reviews
            $reviews = Review::where('user_id', auth()->id())
                ->with(['location', 'drafts'])
                ->orderBy('created_at', 'desc')
                ->paginate(20);

            $googleConnectionsQuery = OauthConnection::where('user_id', auth()->id())
                ->where('provider', 'google');
            if (auth()->user()?->tenant_id) {
                $googleConnectionsQuery->where('tenant_id', auth()->user()->tenant_id);
            } else {
                $googleConnectionsQuery->whereNull('tenant_id');
            }
            $googleConnections = $googleConnectionsQuery->get();
            $aiManager = app(AIServiceManager::class);
            $providerInfo = $aiManager->getProviderInfo();
            $defaultKey = collect($providerInfo)->filter(fn ($info) => $info['is_default'])->keys()->first();
            $defaultProvider = $defaultKey ? $providerInfo[$defaultKey] : null;

            $locationsQuery = Location::where('user_id', auth()->id());
            if (auth()->user()?->tenant_id) {
                $locationsQuery->where('tenant_id', auth()->user()->tenant_id);
            } else {
                $locationsQuery->whereNull('tenant_id');
            }
            $locations = $locationsQuery->get();

            $status = [
                'oauth_connected' => $googleConnections->isNotEmpty(),
                'oauth_emails' => $googleConnections->pluck('account_email')->filter()->values()->all(),
                'locations_count' => $locations->count(),
                'reviews_count' => $reviews->total(),
                'ai_default_name' => $defaultProvider['name'] ?? 'Not set',
                'ai_default_ready' => $defaultProvider['configured'] ?? false,
            ];
                
            return view('agent.reviews.index', compact('reviews', 'status'));
        } catch (\Exception $e) {
            Log::error('Agent reviews index error: ' . $e->getMessage());
            return view('agent.reviews.index', ['reviews' => collect()]);
        }
    }

    public function disconnectGoogle()
    {
        $user = Auth::user();

        if (!$user) {
            return redirect()->route('agent.reviews.index')
                ->with('error', 'You must be logged in to disconnect Google Business Profile.');
        }

        DB::transaction(function () use ($user) {
            $reviewsQuery = Review::where('user_id', $user->id)
                ->where('tenant_id', $user->tenant_id);

            $reviewIds = $reviewsQuery->pluck('id');
            ReviewDraft::whereIn('review_id', $reviewIds)->delete();
            $reviewsQuery->delete();

            Location::where('user_id', $user->id)
                ->where('tenant_id', $user->tenant_id)
                ->delete();

            OauthConnection::where('provider', 'google')
                ->where('user_id', $user->id)
                ->where('tenant_id', $user->tenant_id)
                ->delete();
        });

        return redirect()->route('agent.reviews.index')
            ->with('success', 'Disconnected Google Business Profile and removed associated data.');
    }


    /**
     * Import locations - Agent can only import their own locations
     */
    public function importLocations()
    {
        try {
            $user = Auth::user();
            $importedCount = 0;
            
            // Check if user has any locations first
            $existingLocationsQuery = Location::where('user_id', $user->id);
            if ($user->tenant_id) {
                $existingLocationsQuery->where('tenant_id', $user->tenant_id);
            } else {
                $existingLocationsQuery->whereNull('tenant_id');
            }
            $existingLocations = $existingLocationsQuery->count();
            if ($existingLocations === 0) {
                return redirect()->route('agent.reviews.index')
                    ->with('error', 'No Google Business Profile locations found for your account. Please connect your Google Business Profile first.');
            }
            
            $gbpService = app(GoogleBusinessProfile::class);
            $gbpLocations = $gbpService->listLocations();
            
            foreach ($gbpLocations as $gbpLocation) {
                Location::updateOrCreate(
                    [
                        'user_id' => $user->id,
                        'tenant_id' => $user->tenant_id,
                        'account_id' => $gbpLocation['account_id'],
                        'location_id' => $gbpLocation['location_id'],
                    ],
                    [
                        'source' => 'google_gbp',
                        'display_name' => $gbpLocation['display_name'],
                    ]
                );
                $importedCount++;
            }

            Log::info('Agent reviews: imported locations', [
                'user_id' => $user->id,
                'tenant_id' => $user->tenant_id,
                'count' => $importedCount,
            ]);
            
            return redirect()->route('agent.reviews.index')
                ->with('success', "Successfully imported {$importedCount} Google Business Profile locations.");
        } catch (\Exception $e) {
            Log::error('Failed to import locations: ' . $e->getMessage());
            return redirect()->route('agent.reviews.index')
                ->with('error', 'Failed to import locations: ' . $e->getMessage());
        }
    }

    /**
     * Ingest reviews - Agent can only ingest reviews from their own locations
     */
    public function ingest()
    {
        try {
            $user = Auth::user();
            
            // Agent sees only their own locations
            $locationsQuery = Location::where('user_id', $user->id);
            if ($user->tenant_id) {
                $locationsQuery->where('tenant_id', $user->tenant_id);
            } else {
                $locationsQuery->whereNull('tenant_id');
            }
            $locations = $locationsQuery->get();
            
            if ($locations->count() === 0) {
                return redirect()->route('agent.reviews.index')
                    ->with('error', 'No Google Business Profile locations found. Please import locations first.');
            }
            
            $ingestedCount = 0;
            $gbpService = app(GoogleBusinessProfile::class);
            
            foreach ($locations as $location) {
                $accountId = basename($location->account_id);
                $locationId = basename($location->location_id);
                $reviews = $gbpService->listReviews($accountId, $locationId);
                
                foreach ($reviews as $reviewData) {
                    Review::updateOrCreate(
                        [
                            'location_id' => $location->id,
                            'external_review_id' => $reviewData['id'],
                        ],
                        [
                            'user_id' => $user->id,
                            'tenant_id' => $user->tenant_id,
                            'source' => 'google_gbp',
                            'stars' => $reviewData['stars'],
                            'lang' => $reviewData['lang'] ?? 'en',
                            'text' => $reviewData['comment'],
                            'status' => 'new',
                        ]
                    );
                    $ingestedCount++;
                }
            }

            Log::info('Agent reviews: ingested reviews', [
                'user_id' => $user->id,
                'tenant_id' => $user->tenant_id,
                'count' => $ingestedCount,
            ]);
            
            return redirect()->route('agent.reviews.index')
                ->with('success', "Successfully ingested {$ingestedCount} reviews.");
        } catch (\Exception $e) {
            Log::error('Failed to ingest reviews: ' . $e->getMessage());
            return redirect()->route('agent.reviews.index')
                ->with('error', 'Failed to ingest reviews: ' . $e->getMessage());
        }
    }

    /**
     * Generate AI draft for review - Agent can only draft their own reviews
     */
    public function draft(Review $review)
    {
        try {
            // Verify review belongs to agent
            if ($review->user_id !== auth()->id()) {
                abort(403, 'You do not have permission to draft this review.');
            }
            
            $aiManager = app(AIServiceManager::class);
            $draftService = new EnhancedReplyDraftService($aiManager);
            $result = $draftService->makeForReview($review);

            if (!empty($result['meta']['error'])) {
                return redirect()->route('agent.reviews.index')
                    ->with('error', 'AI draft failed: ' . ($result['meta']['message'] ?? 'Provider error'));
            }
            
            $draft = ReviewDraft::create([
                'review_id' => $review->id,
                'user_id' => $review->user_id,
                'tenant_id' => $review->tenant_id,
                'draft_reply' => $result['draft'],
                'ai_meta' => $result['meta'],
                'status' => 'pending',
            ]);
            
            $review->update(['status' => 'drafted']);
            
            return redirect()->route('agent.reviews.index')
                ->with('success', 'AI draft generated successfully');
        } catch (\Exception $e) {
            Log::error('Failed to generate review draft: ' . $e->getMessage());
            return redirect()->route('agent.reviews.index')
                ->with('error', 'Failed to generate draft: ' . $e->getMessage());
        }
    }

    /**
     * Approve and post review reply - Agent can only approve their own reviews
     */
    public function approve(Review $review)
    {
        try {
            // Verify review belongs to agent
            if ($review->user_id !== auth()->id()) {
                abort(403, 'You do not have permission to approve this review.');
            }
            
            $draft = $review->latestDraft();
            if (!$draft) {
                return redirect()->route('agent.reviews.index')
                    ->with('error', 'No draft found for this review');
            }
            
            $gbpService = app(GoogleBusinessProfile::class);
            $result = $gbpService->replyToReview(
                $review->location->account_id,
                $review->location->location_id,
                $review->external_review_id,
                $draft->draft_reply
            );
            
            if ($result['success']) {
                $draft->update(['status' => 'sent']);
                $review->update([
                    'status' => 'posted',
                    'reviewed_at' => now(),
                ]);
                
                return redirect()->route('agent.reviews.index')
                    ->with('success', 'Reply posted successfully');
            } else {
                return redirect()->route('agent.reviews.index')
                    ->with('error', 'Failed to post reply: ' . $result['message']);
            }
        } catch (\Exception $e) {
            Log::error('Failed to post review reply: ' . $e->getMessage());
            return redirect()->route('agent.reviews.index')
                ->with('error', 'Failed to post reply: ' . $e->getMessage());
        }
    }

    /**
     * Bulk delete reviews - Agent can only delete their own reviews
     */
    public function bulkDelete(Request $request)
    {
        try {
            $reviewIds = $request->input('review_ids', []);
            
            if (empty($reviewIds)) {
                return redirect()->route('agent.reviews.index')
                    ->with('error', 'No reviews selected for deletion.');
            }

            // Agent can only delete their own reviews
            $reviewsToDelete = Review::where('user_id', auth()->id())
                ->whereIn('id', $reviewIds)
                ->get();

            if ($reviewsToDelete->count() !== count($reviewIds)) {
                return redirect()->route('agent.reviews.index')
                    ->with('error', 'Some selected reviews could not be found or you do not have permission to delete them.');
            }

            $deletedCount = 0;
            foreach ($reviewsToDelete as $review) {
                // Delete associated drafts first
                $review->drafts()->delete();
                
                // Delete the review
                $review->delete();
                $deletedCount++;
            }

            return redirect()->route('agent.reviews.index')
                ->with('success', "Successfully deleted {$deletedCount} review(s).");

        } catch (\Exception $e) {
            Log::error('Failed to bulk delete reviews: ' . $e->getMessage());
            return redirect()->route('agent.reviews.index')
                ->with('error', 'Failed to delete reviews: ' . $e->getMessage());
        }
    }

    /**
     * Delete individual review - Agent can only delete their own reviews
     */
    public function destroy(Review $review)
    {
        try {
            // Verify review belongs to agent
            if ($review->user_id !== auth()->id()) {
                abort(403, 'You do not have permission to delete this review.');
            }

            // Delete associated drafts first
            $review->drafts()->delete();
            
            // Delete the review
            $review->delete();

            return redirect()->route('agent.reviews.index')
                ->with('success', 'Review deleted successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to delete review: ' . $e->getMessage());
            return redirect()->route('agent.reviews.index')
                ->with('error', 'Failed to delete review: ' . $e->getMessage());
        }
    }
}
