<?php

namespace App\Exceptions;

use Illuminate\Foundation\Exceptions\Handler as ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Throwable;

class Handler extends ExceptionHandler
{
    /**
     * The list of the inputs that are never flashed to the session on validation exceptions.
     *
     * @var array<int, string>
     */
    protected $dontFlash = [
        'current_password',
        'password',
        'password_confirmation',
    ];

    /**
     * Register the exception handling callbacks for the application.
     */
    public function register(): void
    {
        $this->reportable(function (Throwable $e) {
            $this->logError($e);
        });
    }

    /**
     * Render an exception into an HTTP response.
     */
    public function render($request, Throwable $e)
    {
        // Log the error before rendering
        $this->logError($e, $request);

        // Handle specific error types
        if ($e instanceof \Illuminate\Database\QueryException) {
            return $this->handleDatabaseError($e, $request);
        }

        if ($e instanceof \GuzzleHttp\Exception\RequestException) {
            return $this->handleHttpError($e, $request);
        }

        if ($e instanceof \Google\Exception) {
            return $this->handleGoogleApiError($e, $request);
        }

        return parent::render($request, $e);
    }

    /**
     * Log error with comprehensive context
     */
    private function logError(Throwable $e, ?Request $request = null): void
    {
        $context = [
            'exception' => get_class($e),
            'message' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString(),
        ];

        if ($request) {
            $context['request'] = [
                'url' => $request->fullUrl(),
                'method' => $request->method(),
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'user_id' => auth()->id(),
                'input' => $request->except(['password', 'password_confirmation']),
            ];
        }

        // Log with appropriate level
        if ($this->isCriticalError($e)) {
            Log::critical('Critical application error', $context);
        } elseif ($this->isError($e)) {
            Log::error('Application error', $context);
        } else {
            Log::warning('Application warning', $context);
        }
    }

    /**
     * Check if error is critical
     */
    private function isCriticalError(Throwable $e): bool
    {
        $criticalTypes = [
            \Illuminate\Database\QueryException::class,
            \Illuminate\Database\ConnectionException::class,
            \RedisException::class,
            \PDOException::class,
        ];

        return in_array(get_class($e), $criticalTypes) || 
               str_contains($e->getMessage(), 'Connection refused') ||
               str_contains($e->getMessage(), 'Database connection');
    }

    /**
     * Check if error is an error (not warning)
     */
    private function isError(Throwable $e): bool
    {
        return !($e instanceof \Illuminate\Validation\ValidationException) &&
               !($e instanceof \Illuminate\Auth\AuthenticationException) &&
               !($e instanceof \Symfony\Component\HttpKernel\Exception\NotFoundHttpException);
    }

    /**
     * Handle database errors
     */
    private function handleDatabaseError(\Illuminate\Database\QueryException $e, Request $request)
    {
        Log::error('Database error occurred', [
            'sql' => $e->getSql(),
            'bindings' => $e->getBindings(),
            'error' => $e->getMessage(),
        ]);

        if ($request->expectsJson()) {
            return response()->json([
                'error' => 'Database error occurred. Please try again later.',
                'code' => 'DATABASE_ERROR'
            ], 500);
        }

        return redirect()->back()->with('error', 'A database error occurred. Please try again later.');
    }

    /**
     * Handle HTTP request errors
     */
    private function handleHttpError(\GuzzleHttp\Exception\RequestException $e, Request $request)
    {
        Log::error('HTTP request failed', [
            'url' => $e->getRequest()->getUri(),
            'method' => $e->getRequest()->getMethod(),
            'error' => $e->getMessage(),
        ]);

        if ($request->expectsJson()) {
            return response()->json([
                'error' => 'External service unavailable. Please try again later.',
                'code' => 'HTTP_ERROR'
            ], 503);
        }

        return redirect()->back()->with('error', 'External service is temporarily unavailable. Please try again later.');
    }

    /**
     * Handle Google API errors
     */
    private function handleGoogleApiError(\Google\Exception $e, Request $request)
    {
        Log::error('Google API error', [
            'error' => $e->getMessage(),
            'user_id' => auth()->id(),
        ]);

        if ($request->expectsJson()) {
            return response()->json([
                'error' => 'Google service error. Please reconnect your account.',
                'code' => 'GOOGLE_API_ERROR'
            ], 400);
        }

        return redirect()->back()->with('error', 'Google service error. Please reconnect your account.');
    }
}
