<?php

namespace App\Console\Commands;

use App\Services\AIServiceManager;
use Illuminate\Console\Command;

class TestAIProvidersCommand extends Command
{
    protected $signature = 'ai:test-providers';
    protected $description = 'Test all configured AI providers';

    public function handle()
    {
        $this->info('Testing AI Providers...');
        $this->newLine();

        $aiManager = app(AIServiceManager::class);
        
        // Get provider info
        $providerInfo = $aiManager->getProviderInfo();
        
        $this->table(
            ['Provider', 'Status', 'Configured', 'Default'],
            collect($providerInfo)->map(function ($info, $key) {
                return [
                    $info['name'],
                    $info['available'] ? '✅ Available' : '❌ Not Available',
                    $info['configured'] ? '✅ Configured' : '❌ Not Configured',
                    $info['is_default'] ? '⭐ Default' : '',
                ];
            })
        );

        $this->newLine();
        
        // Test all providers
        $this->info('Testing provider connections...');
        $results = $aiManager->testAllProviders();
        
        foreach ($results as $provider => $result) {
            if ($result['success']) {
                $this->info("✅ {$provider}: " . $result['message']);
                if (isset($result['response'])) {
                    $this->line("   Response: " . $result['response']);
                }
            } else {
                $this->error("❌ {$provider}: " . ($result['error'] ?? 'Unknown error'));
            }
        }

        $this->newLine();
        
        // Test enhanced reply service
        $this->info('Testing Enhanced Reply Service...');
        try {
            $enhancedService = app(\App\Services\EnhancedReplyDraftService::class);
            $defaultProvider = $aiManager->getDefaultProvider();
            $this->info("Default provider: " . $defaultProvider->getProviderName());
        } catch (\Exception $e) {
            $this->error("Failed to initialize Enhanced Reply Service: " . $e->getMessage());
        }

        $this->newLine();
        $this->info('AI Provider testing completed!');
    }
}
