<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Schedule;
use Illuminate\Support\Facades\Artisan;

class ScheduleStatus extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'schedule:status {--test : Test run scheduled commands}';

    /**
     * The console command description.
     */
    protected $description = 'Show status of all scheduled tasks and optionally test them';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('📅 Scheduled Tasks Status');
        $this->newLine();

        $scheduledTasks = [
            [
                'name' => 'Email Cleanup',
                'command' => 'email:cleanup-old',
                'schedule' => 'Daily at 2:00 AM',
                'description' => 'Clean up old ingested emails',
                'status' => 'Active'
            ],
            [
                'name' => 'Email Auto-Ingest',
                'command' => 'email:auto-ingest',
                'schedule' => 'Every 15 minutes',
                'description' => 'Automatically ingest emails from all accounts',
                'status' => 'Active'
            ],
            [
                'name' => 'Reviews Auto-Ingest',
                'command' => 'reviews:auto-ingest',
                'schedule' => 'Every hour',
                'description' => 'Automatically ingest reviews from all business profiles',
                'status' => 'Active'
            ],
            [
                'name' => 'Production Optimization',
                'command' => 'production:optimize',
                'schedule' => 'Manual only',
                'description' => 'Optimize database and cache',
                'status' => 'Manual'
            ],
            [
                'name' => 'Security Audit',
                'command' => 'security:audit',
                'schedule' => 'Manual only',
                'description' => 'Comprehensive security audit',
                'status' => 'Manual'
            ]
        ];

        // Display table
        $this->table(
            ['Task', 'Command', 'Schedule', 'Description', 'Status'],
            array_map(function($task) {
                return [
                    $task['name'],
                    $task['command'],
                    $task['schedule'],
                    $task['description'],
                    $task['status']
                ];
            }, $scheduledTasks)
        );

        $this->newLine();
        $this->info('🔄 Queue Status:');
        
        // Check if queue worker is running
        $this->checkQueueStatus();

        if ($this->option('test')) {
            $this->newLine();
            $this->info('🧪 Testing scheduled commands...');
            $this->testScheduledCommands();
        }

        $this->newLine();
        $this->info('📋 Management Commands:');
        $this->line('  • Start scheduler: php artisan schedule:work');
        $this->line('  • Run once: php artisan schedule:run');
        $this->line('  • List tasks: php artisan schedule:list');
        $this->line('  • Start queue: php artisan queue:work');
        $this->line('  • Test ingestion: php artisan email:auto-ingest');
        $this->line('  • Test reviews: php artisan reviews:auto-ingest');

        return Command::SUCCESS;
    }

    /**
     * Check queue worker status
     */
    private function checkQueueStatus(): void
    {
        try {
            // Try to get queue status
            $output = Artisan::call('queue:monitor');
            $this->info('  ✅ Queue system is available');
        } catch (\Exception $e) {
            $this->warn('  ⚠️  Queue system may not be running');
            $this->line('     Start with: php artisan queue:work');
        }
    }

    /**
     * Test scheduled commands
     */
    private function testScheduledCommands(): void
    {
        $commands = [
            'email:auto-ingest' => 'Email Auto-Ingest',
            'reviews:auto-ingest' => 'Reviews Auto-Ingest',
            'email:cleanup-old --retention=1' => 'Email Cleanup (test)',
        ];

        foreach ($commands as $command => $name) {
            $this->info("  Testing {$name}...");
            
            try {
                $exitCode = Artisan::call($command);
                if ($exitCode === 0) {
                    $this->info("    ✅ {$name} completed successfully");
                } else {
                    $this->warn("    ⚠️  {$name} completed with warnings");
                }
            } catch (\Exception $e) {
                $this->error("    ❌ {$name} failed: " . $e->getMessage());
            }
        }
    }
}
