<?php

namespace App\Console\Commands;

use App\Services\CacheService;
use App\Services\DatabaseOptimizationService;
use App\Services\AuditLogService;
use Illuminate\Console\Command;

class ProductionOptimize extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'production:optimize {--warm-cache : Warm up cache} {--optimize-db : Optimize database} {--cleanup-audit : Cleanup audit logs}';

    /**
     * The console command description.
     */
    protected $description = 'Run production optimization tasks';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting production optimization...');

        if ($this->option('warm-cache')) {
            $this->warmUpCache();
        }

        if ($this->option('optimize-db')) {
            $this->optimizeDatabase();
        }

        if ($this->option('cleanup-audit')) {
            $this->cleanupAuditLogs();
        }

        $this->info('Production optimization completed!');
        return Command::SUCCESS;
    }

    /**
     * Warm up cache
     */
    private function warmUpCache(): void
    {
        $this->info('Warming up cache...');
        
        try {
            CacheService::warmUpCache();
            $this->info('✅ Cache warmed up successfully');
        } catch (\Exception $e) {
            $this->error('❌ Cache warm up failed: ' . $e->getMessage());
        }
    }

    /**
     * Optimize database
     */
    private function optimizeDatabase(): void
    {
        $this->info('Optimizing database...');
        
        try {
            // Optimize tables
            $optimizeResults = DatabaseOptimizationService::optimizeTables();
            foreach ($optimizeResults as $table => $result) {
                $this->line("  {$table}: {$result}");
            }

            // Analyze indexes
            $indexResults = DatabaseOptimizationService::analyzeIndexes();
            foreach ($indexResults as $table => $data) {
                if (!empty($data['missing_indexes'])) {
                    $this->warn("  {$table}: Missing indexes - " . implode(', ', $data['missing_indexes']));
                }
            }

            // Clean up old data
            $cleanupResults = DatabaseOptimizationService::cleanupOldData();
            foreach ($cleanupResults as $type => $result) {
                $this->line("  {$type}: {$result}");
            }

            $this->info('✅ Database optimization completed');
        } catch (\Exception $e) {
            $this->error('❌ Database optimization failed: ' . $e->getMessage());
        }
    }

    /**
     * Cleanup audit logs
     */
    private function cleanupAuditLogs(): void
    {
        $this->info('Cleaning up audit logs...');
        
        try {
            $deletedCount = AuditLogService::cleanupOldLogs();
            $this->info("✅ Cleaned up {$deletedCount} old audit log entries");
        } catch (\Exception $e) {
            $this->error('❌ Audit log cleanup failed: ' . $e->getMessage());
        }
    }
}
