<?php

namespace App\Console\Commands;

use App\Models\Location;
use App\Jobs\IngestReviewsJob;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

/**
 * Ingest All Reviews Command
 *
 * Dispatches review ingestion jobs for all Google Business Profile locations in the system.
 * Scheduled to run every 15 minutes via Laravel Task Scheduler.
 *
 * Reference: AUTO_INGESTION.md Section 3.3
 */
class IngestAllReviewsCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'reviews:ingest-all';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Dispatch review ingestion jobs for all Google Business Profile locations';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('Starting review ingestion for all locations...');

        // Get all locations (bypassing global scopes to get ALL locations system-wide)
        // Each job is scoped to a specific location, ensuring multi-tenant safety
        $locations = Location::withoutGlobalScopes()->with('tenant')->get();

        if ($locations->count() === 0) {
            $this->warn('No Google Business Profile locations found in the system.');
            Log::warning('IngestAllReviewsCommand: No locations found');
            return Command::SUCCESS;
        }

        $this->info("Found {$locations->count()} location(s). Checking tenant intervals...");

        $jobsDispatched = 0;
        $jobsSkipped = 0;

        // Group locations by tenant to check intervals
        $locationsByTenant = $locations->groupBy('tenant_id');

        foreach ($locationsByTenant as $tenantId => $tenantLocations) {
            $tenant = $tenantLocations->first()->tenant;

            // Get tenant's configured interval (default: 60 minutes)
            $interval = $tenant->settings['review_ingestion_interval'] ?? 60;

            // Check if enough time has passed since last ingestion
            $cacheKey = "review_ingestion_last_run_tenant_{$tenantId}";
            $lastRun = \Illuminate\Support\Facades\Cache::get($cacheKey);

            if ($lastRun && now()->diffInMinutes($lastRun) < $interval) {
                $this->line("⏭ Skipping tenant {$tenant->name}: interval not reached ({$interval} min)");
                $jobsSkipped += $tenantLocations->count();
                continue;
            }

            // Dispatch jobs for this tenant's locations
            foreach ($tenantLocations as $location) {
                try {
                    IngestReviewsJob::dispatch($location);
                    $jobsDispatched++;
                    $this->line("✓ Dispatched job for: {$location->display_name}");

                } catch (\Exception $e) {
                    $this->error("✗ Failed to dispatch job for {$location->display_name}: {$e->getMessage()}");
                    Log::error('IngestAllReviewsCommand: Failed to dispatch job', [
                        'location_name' => $location->display_name,
                        'error' => $e->getMessage()
                    ]);
                }
            }

            // Update last run timestamp for this tenant
            \Illuminate\Support\Facades\Cache::put($cacheKey, now(), now()->addDay());
        }

        $this->info("Dispatched {$jobsDispatched} job(s), skipped {$jobsSkipped} (interval not reached).");

        Log::info('IngestAllReviewsCommand: Completed', [
            'total_locations' => $locations->count(),
            'jobs_dispatched' => $jobsDispatched,
            'jobs_skipped' => $jobsSkipped
        ]);

        return Command::SUCCESS;
    }
}
