<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Thread;
use App\Models\Message;
use App\Models\EmailDraft;
use Carbon\Carbon;

class CleanupOldEmails extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'email:cleanup-old {--retention=24 : Hours to retain emails (default: 24)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Clean up old ingested emails older than specified retention period';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $retentionHours = (int) $this->option('retention');
        $cutoffTime = Carbon::now()->subHours($retentionHours);
        
        $this->info("Starting email cleanup - removing emails older than {$retentionHours} hours");
        $this->info("Cutoff time: {$cutoffTime->format('Y-m-d H:i:s')}");
        
        // Clean up threads without drafts or with only old drafts
        $deletedCount = $this->cleanupOldThreads($cutoffTime);
        
        // Clean up orphaned messages
        $orphanedMessages = $this->cleanupOrphanedMessages();
        
        $totalDeleted = $deletedCount + $orphanedMessages;
        
        $this->info("Cleanup completed!");
        $this->info("Threads deleted: {$deletedCount}");
        $this->info("Orphaned messages deleted: {$orphanedMessages}");
        $this->info("Total records deleted: {$totalDeleted}");
        
        return Command::SUCCESS;
    }
    
    /**
     * Clean up old threads that meet cleanup criteria
     */
    private function cleanupOldThreads(Carbon $cutoffTime): int
    {
        // Find threads older than cutoff time that:
        // 1. Have no active drafts (no drafts or only old drafts)
        // 2. Have not been interacted with recently
        $threadsToDelete = Thread::where('created_at', '<', $cutoffTime)
            ->whereDoesntHave('emailDrafts', function($query) use ($cutoffTime) {
                $query->where('created_at', '>=', $cutoffTime)
                      ->orWhere('updated_at', '>=', $cutoffTime);
            })
            ->get();
        
        $deletedCount = 0;
        
        foreach ($threadsToDelete as $thread) {
            $this->info("Deleting thread: {$thread->subject} (ID: {$thread->id})");
            
            // Delete related drafts first
            EmailDraft::where('thread_id', $thread->id)->delete();
            
            // Delete related messages
            Message::where('thread_id', $thread->id)->delete();
            
            // Delete the thread
            $thread->delete();
            
            $deletedCount++;
        }
        
        return $deletedCount;
    }
    
    /**
     * Clean up orphaned messages (messages without threads)
     */
    private function cleanupOrphanedMessages(): int
    {
        $orphanedMessages = Message::whereDoesntHave('thread')->count();
        
        if ($orphanedMessages > 0) {
            $this->info("Found {$orphanedMessages} orphaned messages - deleting them");
            Message::whereDoesntHave('thread')->delete();
        }
        
        return $orphanedMessages;
    }
}
