<?php

namespace App\Console\Commands;

use App\Jobs\ProcessReviewIngestion;
use App\Models\Location;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class AutoIngestReviews extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'reviews:auto-ingest {--business= : Specific business profile ID to process}';

    /**
     * The console command description.
     */
    protected $description = 'Automatically ingest reviews from all configured business profiles';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔄 Starting automatic review ingestion...');
        
        $businessId = $this->option('business');
        
        // Get business profiles to process
        if ($businessId) {
            $business = Location::find($businessId);
            if (!$business) {
                $this->error("Location with ID " . e($businessId) . " not found.");
                return Command::FAILURE;
            }
            $businesses = collect([$business]);
        } else {
            $businesses = Location::all();
        }

        if ($businesses->isEmpty()) {
            $this->warn('No locations found for review ingestion.');
            return Command::SUCCESS;
        }

        $processedCount = 0;
        $errorCount = 0;

        foreach ($businesses as $business) {
            try {
                $this->info("Processing location: " . e($business->display_name));
                
                // Check if location has required configuration
                if (!$business->location_id) {
                    $this->warn("  ⚠️  Skipping - No location ID found");
                    continue;
                }
                
                // Queue the review ingestion job
                ProcessReviewIngestion::dispatch($business);
                $this->info("  ✅ Queued review ingestion");
                
                $processedCount++;
                
                // Log the auto-ingestion
                Log::info('Auto review ingestion queued', [
                    'location_id' => $business->id,
                    'display_name' => $business->display_name,
                    'scheduled_at' => now(),
                ]);
                
            } catch (\Exception $e) {
                $this->error("  ❌ Error processing " . e($business->display_name) . ": " . e($e->getMessage()));
                $errorCount++;
                
                Log::error('Auto review ingestion failed', [
                    'location_id' => $business->id,
                    'display_name' => $business->display_name,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        $this->newLine();
        $this->info("📊 Auto-ingestion summary:");
        $this->info("  ✅ Businesses processed: {$processedCount}");
        $this->info("  ❌ Errors: {$errorCount}");
        $this->info("  📝 Total jobs queued: {$processedCount}");
        
        if ($processedCount > 0) {
            $this->info("  🔄 Jobs are running in the background. Check queue status with: php artisan queue:work");
        }

        return Command::SUCCESS;
    }
}
