<?php

namespace App\Console\Commands;

use App\Jobs\ProcessEmailIngestion;
use App\Models\MailAccount;
use App\Models\User;
use App\Services\UserContextService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class AutoIngestEmails extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'email:auto-ingest {--user= : Specific user ID to process} {--account= : Specific account ID to process} {--type=all : Ingestion type (imap|oauth|all)}';

    /**
     * The console command description.
     */
    protected $description = 'Automatically ingest emails from all configured accounts';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔄 Starting automatic email ingestion...');
        
        $userId = $this->option('user');
        $accountId = $this->option('account');
        $ingestionType = $this->option('type');
        
        // Get users to process
        if ($userId) {
            $user = User::find($userId);
            if (!$user) {
                $this->error("User with ID " . e($userId) . " not found.");
                return Command::FAILURE;
            }
            $users = collect([$user]);
        } else {
            $users = User::where('is_active', true)->get();
        }

        $totalProcessed = 0;
        $totalErrors = 0;

        foreach ($users as $user) {
            $this->info("Processing user: " . e($user->name) . " (" . e($user->email) . ")");
            
            // Get user's mail accounts
            $userContext = new UserContextService();
            $accounts = $userContext->getUserMailAccounts()->get();
            
            if ($accountId) {
                $account = $accounts->find($accountId);
                if (!$account) {
                    $this->error("Account with ID " . e($accountId) . " not found for user " . e($user->name) . ".");
                    continue;
                }
                $accounts = collect([$account]);
            }

            $processedCount = 0;
            $errorCount = 0;

            foreach ($accounts as $account) {
                try {
                    $this->info("Processing account: " . e($account->email));
                    
                    // Check if account has the required configuration
                    $canProcess = false;
                    $processType = null;
                    
                    if ($ingestionType === 'all' || $ingestionType === 'imap') {
                        if ($account->hasImapConfig()) {
                            $canProcess = true;
                            $processType = 'imap';
                        }
                    }
                    
                    if ($ingestionType === 'all' || $ingestionType === 'oauth') {
                        if ($account->oauth_connection_id) {
                            $canProcess = true;
                            $processType = $processType ? 'both' : 'oauth';
                        }
                    }
                    
                    if (!$canProcess) {
                        $this->warn("  ⚠️  Skipping - No valid configuration for " . e($ingestionType) . " ingestion");
                        continue;
                    }
                    
                    // Queue the ingestion job
                    if ($processType === 'both') {
                        // Queue both IMAP and OAuth if both are available
                        ProcessEmailIngestion::dispatch($account, 'imap');
                        ProcessEmailIngestion::dispatch($account, 'oauth');
                        $this->info("  ✅ Queued both IMAP and OAuth ingestion");
                    } else {
                        ProcessEmailIngestion::dispatch($account, $processType);
                        $this->info("  ✅ Queued " . e($processType) . " ingestion");
                    }
                    
                    $processedCount++;
                    
                    // Log the auto-ingestion
                    Log::info('Auto email ingestion queued', [
                        'account_id' => $account->id,
                        'account_email' => $account->email,
                        'ingestion_type' => $processType,
                        'scheduled_at' => now(),
                    ]);
                    
                } catch (\Exception $e) {
                    $this->error("  ❌ Error processing " . e($account->email) . ": " . e($e->getMessage()));
                    $errorCount++;
                    
                    Log::error('Auto email ingestion failed', [
                        'account_id' => $account->id,
                        'account_email' => $account->email,
                        'error' => $e->getMessage(),
                    ]);
                }
            }
            
            $this->info("  📊 User summary: {$processedCount} accounts processed, {$errorCount} errors");
            $totalProcessed += $processedCount;
            $totalErrors += $errorCount;
        }

        $this->newLine();
        $this->info("📊 Overall Auto-ingestion summary:");
        $this->info("  ✅ Total accounts processed: {$totalProcessed}");
        $this->info("  ❌ Total errors: {$totalErrors}");
        $this->info("  📧 Total jobs queued: " . ($totalProcessed * ($ingestionType === 'all' ? 2 : 1)));
        
        if ($totalProcessed > 0) {
            $this->info("  🔄 Jobs are running in the background. Check queue status with: php artisan queue:work");
        }

        return Command::SUCCESS;
    }
}
