<?php

namespace App\Console\Commands;

use App\Models\User;
use App\Models\Tenant;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class AppInstallCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:install';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Install ReplyPilot application';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🚀 Installing ReplyPilot...');
        
        // Run migrations
        $this->info('📦 Running database migrations...');
        $this->call('migrate');
        
        // Create admin user
        $this->info('👤 Creating admin user...');
        $this->createAdminUser();
        
        // Display next steps
        $this->displayNextSteps();
        
        $this->info('✅ ReplyPilot installation completed successfully!');
    }
    
    /**
     * Create admin user
     */
    private function createAdminUser()
    {
        // First, create a default tenant
        $tenantName = $this->ask('Organization/Tenant name', config('app.name', 'ReplyPilot'));

        $tenant = Tenant::firstOrCreate(
            ['slug' => Str::slug($tenantName)],
            [
                'name' => $tenantName,
                'slug' => Str::slug($tenantName),
                'is_active' => true,
            ]
        );

        $this->info("Tenant '{$tenantName}' created/found.");

        $name = $this->ask('Admin user name');
        $email = $this->ask('Admin email address');
        $password = $this->secret('Admin password (min 8 characters)');

        // Validate password
        if (strlen($password) < 8) {
            $this->error('Password must be at least 8 characters long.');
            return $this->createAdminUser();
        }

        // Validate email
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->error('Please enter a valid email address.');
            return $this->createAdminUser();
        }

        // Check if user already exists
        if (User::where('email', $email)->exists()) {
            $this->error("A user with email '{$email}' already exists.");
            return $this->createAdminUser();
        }

        try {
            User::create([
                'name' => $name,
                'email' => $email,
                'password' => Hash::make($password),
                'role' => 'super_admin',
                'tenant_id' => $tenant->id,
            ]);

            $this->info("Super admin user '{$name}' created successfully!");
        } catch (\Exception $e) {
            $this->error('Failed to create admin user: ' . $e->getMessage());
        }
    }
    
    /**
     * Display next steps
     */
    private function displayNextSteps()
    {
        $this->info('');
        $this->info('📋 Next Steps:');
        $this->info('');
        $this->info('1. Configure your .env file with:');
        $this->info('   - OPENAI_API_KEY=your_openai_api_key');
        $this->info('   - GOOGLE_CLIENT_ID=your_google_client_id');
        $this->info('   - GOOGLE_CLIENT_SECRET=your_google_client_secret');
        $this->info('   - GOOGLE_REDIRECT_URI=' . config('app.url') . '/oauth/google/callback');
        $this->info('   - GOOGLE_ALLOWED_SCOPES=https://www.googleapis.com/auth/gmail.readonly,https://www.googleapis.com/auth/gmail.send,https://www.googleapis.com/auth/business.manage');
        $this->info('');
        $this->info('2. Start the queue worker:');
        $this->info('   php artisan queue:work');
        $this->info('');
        $this->info('3. Set up cron job for scheduler:');
        $this->info('   * * * * * cd ' . base_path() . ' && php artisan schedule:run >> /dev/null 2>&1');
        $this->info('');
        $this->info('4. Visit your application:');
        $this->info('   ' . config('app.url'));
        $this->info('');
        $this->info('5. Connect your Google account and start managing reviews and emails!');
    }
}
