# Google Business Profile API Setup Guide

This guide documents the complete process of setting up Google Business Profile API integration for ReplyPilot.

## 📋 Table of Contents

- [Overview](#overview)
- [Prerequisites](#prerequisites)
- [Required APIs](#required-apis)
- [Google Cloud Console Setup](#google-cloud-console-setup)
- [API Access Request](#api-access-request)
- [OAuth Configuration](#oauth-configuration)
- [Implementation](#implementation)
- [Database Setup](#database-setup)
- [Troubleshooting](#troubleshooting)
- [Testing](#testing)

## 🎯 Overview

The Google Business Profile API integration enables ReplyPilot to:

- ✅ Fetch real business locations from Google Business Profile
- ✅ Retrieve actual customer reviews
- ✅ Generate AI-powered responses to reviews
- ✅ Post approved responses back to Google Business Profile

## 📋 Prerequisites

Before starting, ensure you have:

- [ ] Google Cloud Console account
- [ ] Active Google Business Profile
- [ ] Laravel application with database access
- [ ] Admin access to your Google Business Profile

## 🔌 Required APIs

You need to enable **8 different APIs** in Google Cloud Console:

### Core APIs (Required)
- **Google My Business API** (`mybusiness.googleapis.com`)
- **My Business Account Management API** (`mybusinessaccountmanagement.googleapis.com`)
- **My Business Business Information API** (`mybusinessbusinessinformation.googleapis.com`)

### Additional APIs (Recommended)
- **My Business Lodging API** (`mybusinesslodging.googleapis.com`)
- **My Business Place Actions API** (`mybusinessplaceactions.googleapis.com`)
- **My Business Notifications API** (`mybusinessnotifications.googleapis.com`)
- **My Business Verifications API** (`mybusinessverifications.googleapis.com`)
- **My Business Q&A API** (`mybusinessqanda.googleapis.com`)

## 🛠️ Google Cloud Console Setup

### Step 1: Create/Select Project

1. Go to [Google Cloud Console](https://console.cloud.google.com/)
2. Create a new project or select existing one
3. Note your Project ID (e.g., `797477578261`)

### Step 2: Enable APIs

For each API listed above:

1. Navigate to [API Library](https://console.developers.google.com/apis/library)
2. Search for the API name (e.g., "Google My Business API")
3. Click on the API
4. Click **"Enable API"**
5. Repeat for all 8 APIs

**Direct Links:**
- [Google My Business API](https://console.developers.google.com/apis/api/mybusiness.googleapis.com/overview)
- [My Business Account Management API](https://console.developers.google.com/apis/api/mybusinessaccountmanagement.googleapis.com/overview)
- [My Business Business Information API](https://console.developers.google.com/apis/api/mybusinessbusinessinformation.googleapis.com/overview)

### Step 3: Verify API Status

After enabling, each API should show:
- **Service name**: `[api-name].googleapis.com`
- **Type**: `Public API`
- **Status**: `Enabled`

## 📝 API Access Request

**⚠️ CRITICAL**: Google Business Profile APIs require special approval!

### Step 1: Submit Access Request

1. Go to [Access Request Form](https://docs.google.com/forms/d/e/1FAIpQLSfC_FKSWzbSae_5rOpgwFeIUzXUF1JCQnlsZM_gC1I2UHjA3w/viewform)
2. Fill out all required fields
3. **Important**: Include email addresses associated with your project
4. Submit the form

### Step 2: Wait for Approval

- Approval process can take **days to weeks**
- You'll receive an email when approved
- Email will contain: "Congratulations! Your project has been approved to use the Google Business Profile API!"

### Step 3: Post-Approval Setup

After approval:
1. APIs will become fully functional
2. You can make real API calls
3. Test with actual business data

## 🔐 OAuth Configuration

### Step 1: Create OAuth Credentials

1. Go to [Credentials](https://console.developers.google.com/apis/credentials)
2. Click **"Create credentials"** → **"OAuth client ID"**
3. Choose **"Web application"**
4. Add redirect URI: `https://developers.google.com/oauthplayground`
5. Click **"Create"**
6. Note down **Client ID** and **Client Secret**

### Step 2: Configure OAuth Scopes

Required scopes:
```
https://www.googleapis.com/auth/business.manage
https://www.googleapis.com/auth/gmail.readonly
https://www.googleapis.com/auth/gmail.send
```

### Step 3: Environment Configuration

Add to your `.env` file:
```env
GOOGLE_CLIENT_ID=your_client_id_here
GOOGLE_CLIENT_SECRET=your_client_secret_here
GOOGLE_REDIRECT_URI=https://developers.google.com/oauthplayground
GOOGLE_ALLOWED_SCOPES=https://www.googleapis.com/auth/gmail.readonly,https://www.googleapis.com/auth/gmail.send,https://www.googleapis.com/auth/business.manage
```

## 💻 Implementation

### API Endpoints Used

#### 1. List Accounts
```http
GET https://mybusinessaccountmanagement.googleapis.com/v1/accounts
Authorization: Bearer <access_token>
```

#### 2. List Locations
```http
GET https://mybusinessbusinessinformation.googleapis.com/v1/accounts/{accountId}/locations?readMask=name,title,websiteUri,storeCode,metadata
Authorization: Bearer <access_token>
```

#### 3. List Reviews
```http
GET https://mybusiness.googleapis.com/v4/accounts/{accountId}/locations/{locationId}/reviews
Authorization: Bearer <access_token>
```

### Key Implementation Notes

#### Resource Name Format
- **Accounts**: Use full resource name `accounts/12345678901234567890`
- **Locations**: Use full resource name `locations/4567890123456789012`
- **Reviews**: Extract numeric IDs from resource names

#### Required Parameters
- **readMask**: `name,title,websiteUri,storeCode,metadata` for locations
- **Authorization**: Bearer token from OAuth flow

#### Data Structure Handling
- **Star Ratings**: Convert text ("ONE", "FIVE") to integers (1, 5)
- **Character Encoding**: Use `utf8mb4_unicode_ci` for emoji support
- **Language**: Default to 'en' if not provided

## 🗄️ Database Setup

### Reviews Table Migration

```bash
php artisan make:migration add_multi_tenant_columns_to_reviews_table
```

```php
Schema::table('reviews', function (Blueprint $table) {
    $table->uuid('user_id')->nullable()->after('id');
    $table->uuid('tenant_id')->nullable()->after('user_id');
    
    $table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
    $table->foreign('tenant_id')->references('id')->on('tenants')->onDelete('cascade');
    
    $table->index(['user_id', 'tenant_id']);
    $table->index(['tenant_id', 'status']);
});
```

### Review Drafts Table Migration

```bash
php artisan make:migration add_multi_tenant_columns_to_review_drafts_table
```

```php
Schema::table('review_drafts', function (Blueprint $table) {
    $table->uuid('user_id')->nullable()->after('id');
    $table->uuid('tenant_id')->nullable()->after('user_id');
    
    $table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
    $table->foreign('tenant_id')->references('id')->on('tenants')->onDelete('cascade');
    
    $table->index(['user_id', 'tenant_id']);
    $table->index(['tenant_id', 'status']);
});
```

### Character Encoding Fix

```bash
php artisan make:migration fix_reviews_table_collation
```

```php
DB::statement('ALTER TABLE reviews CONVERT TO CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci');
DB::statement('ALTER TABLE reviews MODIFY COLUMN text LONGTEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci');
```

## 🐛 Troubleshooting

### Common Issues & Solutions

#### 1. "API has not been used" Error (403)
**Problem**: 403 error even with APIs enabled
**Solution**: Submit API access request form

#### 2. "Invalid argument" Error (400)
**Problem**: Wrong endpoint or parameters
**Solutions**:
- Use full resource names (`accounts/123...`) not just IDs
- Include `readMask` parameter for locations
- Use correct API endpoints

#### 3. "Unknown column 'user_id'" Error
**Problem**: Missing multi-tenant columns
**Solution**: Run migrations to add `user_id` and `tenant_id` columns

#### 4. "Incorrect integer value: 'ONE'" Error
**Problem**: Text star ratings instead of integers
**Solution**: Convert text ratings ("ONE" → 1, "FIVE" → 5)

#### 5. "Conversion from utf8mb4 to utf8mb3 impossible" Error
**Problem**: Character encoding mismatch with emojis
**Solution**: Update table collation to `utf8mb4_unicode_ci`

#### 6. "Requested entity was not found" Error (404)
**Problem**: Wrong account/location ID format
**Solution**: Use numeric IDs extracted from resource names

## 🧪 Testing

### Step 1: Test API Connection

```php
// Test script to verify API access
$gbpService = app(\App\Services\GoogleBusinessProfile::class);
$locations = $gbpService->listLocations();
echo "Locations found: " . count($locations);
```

### Step 2: Test Review Ingestion

1. Go to `/reviews` in your app
2. Click **"Import Locations"**
3. Click **"Fetch Reviews"**
4. Verify reviews are imported successfully

### Step 3: Verify Data Quality

- [ ] Check that star ratings are integers (1-5)
- [ ] Verify emojis are stored correctly
- [ ] Confirm review text is complete
- [ ] Test AI draft generation

## 📊 Complete Workflow

### Development Phase (Before Approval)
1. Enable all 8 APIs in Google Cloud Console
2. Submit API access request form
3. Implement with test data fallback
4. Set up database schema
5. Configure OAuth credentials

### Production Phase (After Approval)
1. Receive approval email
2. Test with real API calls
3. Verify data ingestion
4. Test AI draft generation
5. Deploy to production

## 🔒 Security Considerations

### API Keys
- Store credentials in environment variables
- Never commit API keys to version control
- Use different keys for development/production

### Data Privacy
- Reviews contain customer information
- Implement proper access controls
- Follow GDPR/privacy regulations
- Log API usage for compliance

### Rate Limiting
- Google APIs have rate limits
- Implement proper error handling
- Use exponential backoff for retries
- Monitor API quota usage

## 📈 Monitoring & Maintenance

### API Quotas
- Monitor daily/monthly quotas
- Set up alerts for quota usage
- Implement caching to reduce API calls

### Error Handling
- Log all API errors
- Implement retry mechanisms
- Monitor for API changes/deprecations

### Data Sync
- Implement periodic sync jobs
- Handle new reviews automatically
- Update existing reviews if modified

## 📚 Additional Resources

- [Google Business Profile API Documentation](https://developers.google.com/my-business)
- [Basic Setup Guide](https://developers.google.com/my-business/content/basic-setup)
- [API Error Codes](https://developers.google.com/my-business/reference/businessinformation/rest/v1/ErrorCode)
- [OAuth 2.0 Playground](https://developers.google.com/oauthplayground)

## 🤝 Support

If you encounter issues:

1. Check the troubleshooting section above
2. Review Google's API documentation
3. Check Laravel logs for detailed error messages
4. Verify all APIs are enabled and approved

---

**Note**: This documentation covers the complete setup process for Google Business Profile API integration. The key was getting API approval and properly handling the various data formats and character encoding issues.
